package com.bambooclod.epassbase.http;

import android.content.Context;
import android.support.annotation.NonNull;

import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclou.util.RandomUtil;
import com.bambooclou.util.auth.V2SignUtil;

import java.io.IOException;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.Date;

import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.logging.HttpLoggingInterceptor;
import okio.Buffer;

/**
 * Comment: 拦截工具类
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-04
 * Email:wangpei@bamboocloud.cn
 */
public class InterceptorUtil {

    private static final String TAG = "InterceptorUtil";
    private final Charset UTF8 = Charset.forName("UTF-8");

    private Context mContext;

    public InterceptorUtil(Context mContext) {
        this.mContext = mContext;
    }

    /**
     * 日志拦截
     *
     * @return HttpLoggingInterceptor
     */
    public HttpLoggingInterceptor LogInterceptor() {
        return new HttpLoggingInterceptor(new HttpLoggingInterceptor.Logger() {
            @Override
            public void log(String message) {
                LogUtil.getInstance().i("log: " + message);
            }
        }).setLevel(HttpLoggingInterceptor.Level.BODY);//设置打印数据的级别
    }

    /**
     * 拦截请求
     * <p>
     * 主要查看是否签名内容
     * 同时也提供鉴权接口的作用
     *
     * @return Interceptor
     */
    public Interceptor HeaderInterceptor() {
        return new Interceptor() {
            @NonNull
            @Override
            public Response intercept(@NonNull Chain chain) throws IOException {

                //获取鉴权用户
                String operator = (String) SPUtils.get(mContext, InitConfigValue.USER_OPERATOR, InitConfigValue.DEFAULT);
                String randomstr = RandomUtil.getNumRandomString(8);
                long timestamp = System.currentTimeMillis();

                //body签名
                String signKey = (String) SPUtils.get(mContext, InitConfigValue.KEY_SIGN, InitConfigValue.DEFAULT);
                String signContent = V2SignUtil.bodySign(signKey, getRequestBody(chain));
                String authKey = (String) SPUtils.get(mContext, InitConfigValue.KEY_AUTH, InitConfigValue.DEFAULT);


                Request.Builder builder = chain.request()
                        .newBuilder()
                        .addHeader("operator", operator)
                        .addHeader("randomstr", randomstr)
                        .addHeader("timestamp", getDateToString(timestamp))
                        .addHeader("encode", V2SignUtil.encode(operator, authKey
                                , Long.valueOf(getDateToString(timestamp)), randomstr));

                boolean isCheckSession = (boolean) SPUtils.get(mContext, InitConfigValue.BODY_IS_CHECK_SESSION, true);
                LogUtil.getInstance().d("isCheckSession:" + isCheckSession);

                //开启了签名
                if (SPUtils.isSignContent(mContext)) {
                    builder.addHeader("sign", signContent);
                }

                //开启了会话
                if (isCheckSession) {
                    String appID = (String) SPUtils.get(mContext, InitConfigValue.APP_ID, InitConfigValue.DEFAULT);
                    String jwt = (String) SPUtils.get(mContext, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
                    builder.addHeader("jwt", jwt);
                    builder.addHeader("appId", appID);
                }

                Request request = builder.build();
                return chain.proceed(request);
            }
        };
    }

    /**
     * 将时间戳改为日期
     */
    public static String getDateToString(long milSecond) {
        String pattern = "yyyyMMddHHmmss";
        Date date = new Date(milSecond);
        SimpleDateFormat format = new SimpleDateFormat(pattern);
        return format.format(date);
    }

    /**
     * 截取请求中的body
     *
     * @return body
     */
    private String getRequestBody(Interceptor.Chain chain) throws IOException {
        RequestBody requestBody = chain.request().body();
        String body = null;
        if (requestBody != null) {
            Buffer buffer = new Buffer();
            requestBody.writeTo(buffer);
            Charset charset = UTF8;
            MediaType contentType = requestBody.contentType();
            if (contentType != null) {
                charset = contentType.charset(UTF8);
            }

            if (charset != null) {
                body = buffer.readString(charset);
            } else {
                body = "";
            }
        }

        LogUtil.getInstance().i("截取的请求内容:" + body);
        return body;

    }
}
