package com.bambooclod.eaccount3.api;

import android.app.Activity;
import android.content.Context;
import android.util.Base64;

import com.alibaba.fastjson.JSON;
import com.bambooclod.eaccount3.bean.FingerAuthPara;
import com.bambooclod.eaccount3.bean.JwtDecodeData;
import com.bambooclod.eaccount3.biometriclib.BiometricPromptManager;
import com.bambooclod.eaccount3.callback.AuthCallBack;
import com.bambooclod.eaccount3.callback.AuthCallByBindBack;
import com.bambooclod.eaccount3.callback.AuxAuthCallBack;
import com.bambooclod.eaccount3.callback.SingalAuthCallBack;
import com.bambooclod.epassbase.config.HttpConfig;
import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.sp.SPUtils;

/**
 * Comment: 指纹认证
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-14
 * Email:wangpei@bamboocloud.cn
 */
public class FingerAuth {

    private FingerAuth() {
    }

    public static final FingerAuth authRequest = new FingerAuth();

    public static FingerAuth get() {
        return authRequest;
    }

    private AuxAuthCallBack mAuxAuthCallBack;
    private SingalAuthCallBack mSingalAuthCallBack;
    private AuthCallBack mAuthCallBack;
    private AuthCallByBindBack mAuthCallBackByBindBack;

    /**
     * 从JWT中获取用户信息
     *
     * @param jwt JWT
     * @return JwtDecodeData
     */
    private JwtDecodeData getUserInfoFromJWT(String jwt) {
        //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
        String[] forma = jwt.split("\\.");
        String jwtContent = new String(Base64.decode(forma[1], Base64.DEFAULT));
        return JSON.parseObject(jwtContent, JwtDecodeData.class);
    }

    /**
     * 指纹辅助认证
     */
    public void auxFrAuth(final Context context, final AuxAuthCallBack callBack) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        final FingerAuthPara auxAuthPara;
        mAuxAuthCallBack = callBack;
        //使用JWT获取token
        if (!InitConfigValue.DEFAULT.equals(jwt)) {
            String userID = getUserInfoFromJWT(jwt).getUserid();
            FingerAuthPara.UserInfoBean userInfoBean =
                    new FingerAuthPara.UserInfoBean(userID, userID, "");
            //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
            auxAuthPara = new FingerAuthPara(userInfoBean, "");
        } else {
            callBack.error("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }

        checkFr(context, new BiometricPromptManager.OnBiometricIdentifyCallback() {
            @Override
            public void onUsePassword() {
                callBack.error("UsePassword", "使用密码逻辑");
            }

            @Override
            public void onSucceeded() {
                AuthRequest.get().doAuth(context, auxAuthPara, ""
                        , HttpConfig.AuthType.FR, false, new AuthCallBack() {
                            @Override
                            public void doAuthSuccess(String authList, String authedList) {
                                if (callBack != null) {
                                    callBack.auxAuthSuccess();
                                }
                            }

                            @Override
                            public void needBindDevice(String authList) {
                            }

                            @Override
                            public void error(String code, String message) {
                                if (callBack != null) {
                                    callBack.error(code, message);
                                }
                            }
                        });
            }

            @Override
            public void onFailed() {
                if (callBack != null) {
                    callBack.error("-1", "onFailed");
                }
            }

            @Override
            public void onError(int code, String reason) {
                if (callBack != null) {
                    callBack.error(code + "", reason);
                }
            }

            @Override
            public void onCancel() {
                if (callBack != null) {
                    callBack.error("-1", "onCancel");
                }
            }
        });
    }

    /**
     * 检测指纹是否为用户的
     */
    private void checkFr(Context context, BiometricPromptManager.OnBiometricIdentifyCallback callBack) {
        BiometricPromptManager manager = BiometricPromptManager.from((Activity) context);
        //判断是否有硬件支持
        if (!manager.isHardwareDetected()) {
            if (mAuxAuthCallBack != null) {
                mAuxAuthCallBack.error("NoSupport", "当前设备不支持指纹解锁功能");
            }
            if (mSingalAuthCallBack != null) {
                mSingalAuthCallBack.error("NoSupport", "当前设备不支持指纹解锁功能");
            }
            if (mAuthCallBack != null) {
                mAuthCallBack.error("NoSupport", "当前设备不支持指纹解锁功能");
            }

            if (mAuthCallBackByBindBack != null) {
                mAuthCallBackByBindBack.error("NoSupport", "当前设备不支持指纹解锁功能");
            }
        } else {
            //判断是否有指纹信息
            if (!manager.hasEnrolledFingerprints()) {
                if (mAuxAuthCallBack != null) {
                    mAuxAuthCallBack.error("NoFinger", "当前设备暂未设置指纹信息");
                }
                if (mSingalAuthCallBack != null) {
                    mSingalAuthCallBack.error("NoFinger", "当前设备暂未设置指纹信息");
                }
                if (mAuthCallBack != null) {
                    mAuthCallBack.error("NoFinger", "当前设备暂未设置指纹信息");
                }
                if (mAuthCallBackByBindBack != null) {
                    mAuthCallBackByBindBack.error("NoFinger", "当前设备暂未设置指纹信息");
                }
            } else {
                //验证通过
                manager.authenticate(callBack);
            }
        }
    }

    /**
     * 指纹单次认证
     */
    public void frSignalAuth(final Context context, final SingalAuthCallBack callBack) {
        //支持离线认证模式
        final String authType = (String) SPUtils.get(context, InitConfigValue.GLOBAL_AUTH_TYPE
                , InitConfigValue.DEFAULT);

        mSingalAuthCallBack = callBack;

        checkFr(context, new BiometricPromptManager.OnBiometricIdentifyCallback() {
            @Override
            public void onUsePassword() {
                callBack.error("UsePassword", "使用密码逻辑");
            }

            @Override
            public void onSucceeded() {
                if (InitConfigValue.DEFAULT.equals(authType) || HttpConfig.GLOBAL_AUTH_TYPE_ONLINE.equals(authType)) {
                    doSingleFingerSuccessResult(context, callBack);
                } else {
                    if (callBack != null) {
                        callBack.signalAuthSuccess(true);
                    }
                }
            }

            @Override
            public void onFailed() {
                if (callBack != null) {
                    callBack.signalAuthSuccess(false);
                }
            }

            @Override
            public void onError(int code, String reason) {
                if (callBack != null) {
                    callBack.error(code + "", reason);
                }
            }

            @Override
            public void onCancel() {
                if (callBack != null) {
                    callBack.signalAuthSuccess(false);
                }
            }
        });
    }

    /**
     * 指纹认证链认证
     */
    public void frLinkAuth(final Context context, final AuthCallBack callBack) {

        String session = (String) SPUtils.get(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(session)) {
            session = "";
        }

        mAuthCallBack = callBack;

        final String finalSession = session;
        checkFr(context, new BiometricPromptManager.OnBiometricIdentifyCallback() {
            @Override
            public void onUsePassword() {
                callBack.error("UsePassword", "使用密码逻辑");
            }

            @Override
            public void onSucceeded() {
                doLinkFingerSuccessResult(context, finalSession, callBack);
            }

            @Override
            public void onFailed() {
                if (callBack != null) {
                    callBack.error("-1", "onFailed");
                }
            }

            @Override
            public void onError(int code, String reason) {
                if (callBack != null) {
                    callBack.error(code + "", reason);
                }
            }

            @Override
            public void onCancel() {
                if (callBack != null) {
                    callBack.error("-1", "onCancel");
                }
            }
        });
    }

    /**
     * 指纹认证链认证
     */
    public void frLinkAuth(final Context context, final AuthCallByBindBack callBack) {

        String session = (String) SPUtils.get(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(session)) {
            session = "";
        }

        mAuthCallBackByBindBack = callBack;

        final String finalSession = session;
        checkFr(context, new BiometricPromptManager.OnBiometricIdentifyCallback() {
            @Override
            public void onUsePassword() {
                callBack.error("UsePassword", "使用密码逻辑");
            }

            @Override
            public void onSucceeded() {
                doLinkFingerSuccessResult(context, finalSession, mAuthCallBackByBindBack);
            }

            @Override
            public void onFailed() {
                if (callBack != null) {
                    callBack.error("-1", "onFailed");
                }
            }

            @Override
            public void onError(int code, String reason) {
                if (callBack != null) {
                    callBack.error(code + "", reason);
                }
            }

            @Override
            public void onCancel() {
                if (callBack != null) {
                    callBack.error("-1", "onCancel");
                }
            }
        });
    }

    /**
     * 从Session中获取用户信息
     */
    private String getUserInfoFromSession(String session) {
        //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
        String[] format = session.split(":");
        return format[1];
    }


    /**
     * 处理单次指纹认证成功后的逻辑
     */
    private void doSingleFingerSuccessResult(Context context, SingalAuthCallBack callBack) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        FingerAuthPara para;
        FingerAuthPara.UserInfoBean userInfoBean;

        if (InitConfigValue.DEFAULT.equals(jwt)) {
            callBack.error("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        } else {
            //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
            String[] forma = jwt.split("\\.");
            String jwtContent = new String(Base64.decode(forma[1], Base64.DEFAULT));
            JwtDecodeData decodeData = JSON.parseObject(jwtContent, JwtDecodeData.class);

            //单次
            userInfoBean =
                    new FingerAuthPara.UserInfoBean(decodeData.getUserid()
                            , decodeData.getUserid(), "");

            para = new FingerAuthPara(userInfoBean, "");
        }
        AuthRequest.get().commonAuth(context, para, HttpConfig.AuthType.FR, false, callBack);
    }

    /**
     * 处理认证链指纹认证成功后的逻辑
     */
    private void doLinkFingerSuccessResult(Context context, String epSessionId, AuthCallBack callBack) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        FingerAuthPara para;
        FingerAuthPara.UserInfoBean userInfoBean;

        //使用Session获取用户名
        if ("".equals(epSessionId) || InitConfigValue.DEFAULT.equals(epSessionId) || epSessionId == null) {
            callBack.error("NoSession", "无Session参数，发生异常");
            return;
        } else {
            para = new FingerAuthPara("", getUserInfoFromSession(epSessionId));
        }

        AuthRequest.get().doAuth(context, para, epSessionId, HttpConfig.AuthType.FR, false, callBack);
    }

    /**
     * 处理认证链指纹认证成功后的逻辑
     */
    private void doLinkFingerSuccessResult(Context context, String epSessionId, AuthCallByBindBack callBack) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        FingerAuthPara para;
        FingerAuthPara.UserInfoBean userInfoBean;

        //使用Session获取用户名
        if ("".equals(epSessionId) || InitConfigValue.DEFAULT.equals(epSessionId) || epSessionId == null) {
            callBack.error("NoSession", "无Session参数，发生异常");
            return;
        } else {
            para = new FingerAuthPara("", getUserInfoFromSession(epSessionId));
        }

        AuthRequest.get().doAuth(context, para, epSessionId, HttpConfig.AuthType.FR, false, callBack);
    }
}
