package com.bambooclod.eaccount3.api;

import android.content.Context;
import android.text.TextUtils;
import android.util.Base64;

import com.alibaba.fastjson.JSON;
import com.bambooclod.eaccount3.bean.CommonAuthBaseData;
import com.bambooclod.eaccount3.bean.DoAuthBaseData;
import com.bambooclod.eaccount3.bean.DoAuthResponse;
import com.bambooclod.eaccount3.bean.JwtDecodeData;
import com.bambooclod.eaccount3.callback.DoOrCommonAuthCallBack;
import com.bambooclod.eaccount3.http.ApiService;
import com.bambooclod.epassbase.api.EpassUtils;
import com.bambooclod.epassbase.bean.EncrypteData;
import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.http.BaseEncryptObserver;
import com.bambooclod.epassbase.http.BaseObserver;
import com.bambooclod.epassbase.http.RetrofitFactory;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.otp.OtpUtils;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclou.util.auth.V2SignUtil;

import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Comment: 对外认证请求工具类
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-04
 * Email:wangpei@bamboocloud.cn
 */
public class AuthRequest {

    private AuthRequest() {
    }

    public static final AuthRequest authRequest = new AuthRequest();

    public static AuthRequest get() {
        return authRequest;
    }

    /**
     * 统一/单次认证接口
     *
     * @param context       context
     * @param t             具体的请求类型的参数
     * @param isSimpleCheck 是否单次认证
     * @param epSessionId   会话ID
     * @param callBack      回调
     * @param <T>           泛型为authPara
     * @param isBindDevice  是否为绑定设备的请求
     */
    protected <T> void doAuth(final Context context, T t, final boolean isSimpleCheck
            , String epSessionId, String authType, boolean isBindDevice, final DoOrCommonAuthCallBack callBack) {

        //Retrofit
        ApiService apiService = RetrofitFactory.getInstance(context)
                .API().create(ApiService.class);

        //是否内容加密
        boolean isEncrypt = SPUtils.isEncrytContent(context);
        String appID = (String) SPUtils.get(context, InitConfigValue.APP_ID, InitConfigValue.DEFAULT);

        //统一认证
        DoAuthBaseData<T> doAuth;

        //是否为绑定DoAuth
        if (isBindDevice) {
            //3为绑定
            doAuth = new DoAuthBaseData<>(appID, t, authType, epSessionId, "3", EpassUtils.get().getDeviceID(context));
        } else {
            doAuth = new DoAuthBaseData<>(appID, t, authType, epSessionId, EpassUtils.get().getDeviceID(context));

        }
        //单次认证
        CommonAuthBaseData<T> commonAuth = new CommonAuthBaseData<>(t, authType);

        //预请求体
        Observer observer;
        Observable observable;

        if (isEncrypt) {
            String body;

            if (!isSimpleCheck) {
                body = JSON.toJSONString(doAuth);
            } else {
                body = JSON.toJSONString(commonAuth);
            }
            String authKey = (String) SPUtils.get(context, InitConfigValue.KEY_ENCRYPT, InitConfigValue.DEFAULT);
            Object data = new EncrypteData(V2SignUtil.encryt(body, authKey));

            //是否为单次认证
            if (!isSimpleCheck) {
                observable = apiService.commonRequestByEncrypt(ApiService.doAuth, data);
            } else {
                observable = apiService.commonRequestByEncrypt(ApiService.commonAuth, data);
            }

            observer = new BaseEncryptObserver(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    authSuccessHandler(context, body, isSimpleCheck, callBack);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("not.common.device.need.check".equals(code)
                            || "not.common.device.no.check".equals(code)) {
                        if (callBack != null) {
                            DoAuthResponse response = JSON.parseObject(object.toString()
                                    , DoAuthResponse.class);
                            callBack.needBindDevice(response.getDevBindExpectAuthType(), response.getEpsessionId());

                            //这里存储Session，当没有jwt，而又有Session的时候
                            if (!TextUtils.isEmpty(response.getEpsessionId())) {
                                LogUtil.getInstance().d("存储Session");
                                SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, response.getEpsessionId());
                            }
                        }
                    } else if ("dispatchcentre.epsession.not.exist".equals(code)
                            || "dispatchcentre.epsession.state.error".equals(code)) {
                        //这两种情况要清除Session
                        LogUtil.getInstance().d("服务器发生异常，清除SESSION");
                        SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);
                        if (callBack != null) {
                            callBack.error(code, message);
                        }
                    } else {
                        if (callBack != null) {
                            callBack.error(code, message);
                        }
                    }
                }
            };
        } else {
            //是否为单次认证
            if (!isSimpleCheck) {
                observable = apiService.commonRequest(ApiService.doAuth, doAuth);
            } else {
                observable = apiService.commonRequest(ApiService.commonAuth, commonAuth);
            }

            observer = new BaseObserver<Object>(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    authSuccessHandler(context, body, isSimpleCheck, callBack);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("not.common.device.need.check".equals(code)
                            || "not.common.device.no.check".equals(code)) {
                        if (callBack != null) {
                            DoAuthResponse response = JSON.parseObject(object.toString()
                                    , DoAuthResponse.class);
                            callBack.needBindDevice(response.getDevBindExpectAuthType(), response.getEpsessionId());

                            //这里存储Session，当没有jwt，而又有Session的时候
                            if (!TextUtils.isEmpty(response.getEpsessionId())) {
                                LogUtil.getInstance().d("存储Session");
                                SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, response.getEpsessionId());
                            }
                        }
                    } else if ("dispatchcentre.epsession.not.exist".equals(code)
                            || "dispatchcentre.epsession.state.error".equals(code)) {
                        //这两种情况要清除Session
                        LogUtil.getInstance().d("服务器发生异常，清除SESSION");
                        SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);
                        if (callBack != null) {
                            callBack.error(code, message);
                        }
                    } else {
                        if (callBack != null) {
                            callBack.error(code, message);
                        }
                    }
                }
            };
        }


        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
    }

    /**
     * 处理认证成功返回的数据，并回调
     */
    private void authSuccessHandler(Context context, Object body, boolean isSimpleCheck
            , final DoOrCommonAuthCallBack callBack) {
        if (callBack != null) {
            if (isSimpleCheck) {
                callBack.doCommonAuthSuccess(true);
            } else {
                DoAuthResponse response = JSON.parseObject(body.toString(), DoAuthResponse.class);
                if (!"".equals(response.getJwt()) && response.getJwt() != null) {
                    LogUtil.getInstance().d("存储JWT，清除SESSION");
                    //存储Jwt
                    SPUtils.put(context, InitConfigValue.USER_JWT, response.getJwt());
                    //拿到Jwt就清除Session
                    SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);
                } else {
                    //这里存储Session，当没有jwt，而又有Session的时候
                    if (!TextUtils.isEmpty(response.getEpsessionId())) {
                        LogUtil.getInstance().d("存储Session");
                        SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, response.getEpsessionId());
                    }
                }

                //服务器没有OTP，也需要覆盖
                LogUtil.getInstance().w("存储OTPKey!");
                OtpUtils.initOtp(context, response.getOptkey());

                callBack.doAuthSuccess(response.getEpsessionId()
                        , response.getAuthList(), response.getAuthedList());
            }
        }
    }

    /**
     * 为了将页面进行关闭的回调接口
     */
    public interface AuthResultListener {
        void finish();
    }

    /**
     * 从JWT中获取用户信息
     *
     * @param jwt JWT
     * @return JwtDecodeData
     */
    public JwtDecodeData getUserInfoFromJWT(String jwt) {
        //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
        String[] forma = jwt.split("\\.");
        String jwtContent = new String(Base64.decode(forma[1], Base64.DEFAULT));
        return JSON.parseObject(jwtContent, JwtDecodeData.class);
    }
}
