package cn.alphabets.light.model;

import cn.alphabets.light.Constant;
import cn.alphabets.light.Environment;
import cn.alphabets.light.cache.CacheManager;
import cn.alphabets.light.db.mongo.Model;
import cn.alphabets.light.entity.ModCategory;
import cn.alphabets.light.entity.ModFile;
import cn.alphabets.light.entity.ModGroup;
import cn.alphabets.light.entity.ModUser;
import cn.alphabets.light.model.deserializer.DateDeserializer;
import cn.alphabets.light.model.deserializer.LongDeserializer;
import cn.alphabets.light.model.deserializer.ObjectIdDeserializer;
import cn.alphabets.light.model.serializer.DateSerializer;
import cn.alphabets.light.model.serializer.ObjectIdSerializer;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.squareup.javapoet.*;
import org.apache.commons.lang3.text.WordUtils;
import org.bson.Document;
import org.bson.types.ObjectId;

import javax.lang.model.element.Modifier;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Generator
 * Created by lilin on 2016/11/8.
 */
public class Generator {

    private String packageName;

    public Generator(String packageName) {
        this.packageName = packageName;
    }

    /**
     * Generates a POJO source
     *
     * @param domain domain
     * @param schema schema
     */
    public void generate(String domain, List<String> schema) {

        Model model = new Model(domain, Constant.SYSTEM_DB_PREFIX, Constant.SYSTEM_DB_STRUCTURE);

        Document condition = new Document("valid", 1);
        condition.put("schema", new Document("$in", schema));
        List<Document> defines = model.document(condition, Arrays.asList("schema", "items", "type"));

        defines.forEach((define) -> {

            final TypeSpec.Builder builder = this.mainClass(define.getString("schema"), define.getInteger("type"));
            Document items = (Document) define.get("items");

            items.entrySet().forEach(item -> {
                if (!isParentField(item.getKey(), define.getInteger("type")) && !lightReserved.contains(item.getKey())) {
                    this.subClass(builder, item.getKey(), (Document) item.getValue());
                }
            });

            this.write(builder.build());
        });
    }

    /**
     * Generates a POJO source for current app
     */
    public void generate() {
        List<String> schemas = new ArrayList<>();
        CacheManager.INSTANCE.getStructures().forEach(structure -> {
            if (structure.getKind() != 2) {
                schemas.add(structure.getSchema());
            }
        });
        generate(Environment.instance().getAppName(), schemas);
    }

    private boolean isParentField(String name, Integer type) {
        if (type > 0) {
            for (Field field : ((Class) structureType.get(type)).getDeclaredFields()) {
                if (field.getName().equals(name)) {
                    return true;
                }
            }
        }
        return false;
    }

    private void write(TypeSpec type) {
        JavaFile javaFile = JavaFile.builder(this.packageName, type).build();
        try {
            javaFile.writeTo(new File("src/main/java"));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private TypeSpec.Builder mainClass(String className, Integer type) {
        return TypeSpec.classBuilder(Constant.MODEL_PREFIX + WordUtils.capitalize(className))
                .addModifiers(Modifier.PUBLIC)
                .superclass(TypeName.get(structureType.get(type)))
                .addJavadoc("Generated by the Light platform. Do not manually modify the code.\n");
    }

    private void subClass(TypeSpec.Builder spec, String name, Document define) {

        // Add the field
        Type type = regular(define.getString("type"));
        if (type != List.class && type != Object.class) {
            this.append(spec, name, this.type(type));
            return;
        }

        // If the subtype is not defined
        Document items = (Document) define.get("contents");
        if (items == null || items.isEmpty()) {
            this.append(spec, name, this.type(type));
            return;
        }

        this.append(spec, name, this.type(define.getString("type"), WordUtils.capitalize(name)));

        AnnotationSpec annotation = AnnotationSpec.builder(JsonIgnoreProperties.class)
                .addMember("ignoreUnknown", "$L", Boolean.TRUE)
                .build();

        TypeSpec.Builder builder = TypeSpec.classBuilder(WordUtils.capitalize(name))
                .addModifiers(Modifier.PUBLIC, Modifier.FINAL, Modifier.STATIC)
                .superclass(Entity.class)
                .addAnnotation(annotation);

        items.entrySet().forEach(item -> this.subClass(builder, item.getKey(), (Document) item.getValue()));

        spec.addType(builder.build());
    }


    private TypeName type(Type name) {
        return TypeName.get(name);
    }

    private TypeName type(String name, String template) {
        Type type = regular(name);

        if (type == Object.class && template != null) {
            return ClassName.get("", template);
        }

        if (type == List.class && template != null) {
            return ParameterizedTypeName.get(ClassName.get("java.util", "List"), ClassName.get("", template));
        }

        return TypeName.get(type);
    }

    private void append(TypeSpec.Builder builder, String name, TypeName type) {
        builder.addField(this.field(name, type));
        builder.addMethod(this.getter(name, type));
        builder.addMethod(this.setter(name, type));
    }

    private FieldSpec field(String name, TypeName type) {

        FieldSpec.Builder builder = FieldSpec.builder(type, reserved.contains(name) ? name + "_" : name)
                .addModifiers(Modifier.PRIVATE);

        if (reserved.contains(name)) {
            builder.addAnnotation(AnnotationSpec.builder(JsonProperty.class)
                    .addMember("value", "$S", name).build());
        }

        if (type.equals(TypeName.get(Long.class))) {
            builder.addAnnotation(AnnotationSpec.builder(JsonDeserialize.class)
                    .addMember("using", "$T.$L", LongDeserializer.class, "class").build());
        }

        if (type.equals(TypeName.get(ObjectId.class))) {
            builder.addAnnotation(AnnotationSpec.builder(JsonDeserialize.class)
                    .addMember("using", "$T.$L", ObjectIdDeserializer.class, "class").build());
            builder.addAnnotation(AnnotationSpec.builder(JsonSerialize.class)
                    .addMember("using", "$T.$L", ObjectIdSerializer.class, "class").build());
        }
        if (type.equals(TypeName.get(Date.class))) {
            builder.addAnnotation(AnnotationSpec.builder(JsonDeserialize.class)
                    .addMember("using", "$T.$L", DateDeserializer.class, "class").build());
            builder.addAnnotation(AnnotationSpec.builder(JsonSerialize.class)
                    .addMember("using", "$T.$L", DateSerializer.class, "class").build());
        }

        return builder.build();
    }

    private MethodSpec getter(String name, TypeName type) {

        String reservedName = reserved.contains(name) ? name + "_" : name;
        return MethodSpec.methodBuilder("get" + WordUtils.capitalize(reservedName))
                .addModifiers(Modifier.PUBLIC)
                .returns(type)
                .addStatement(String.format("return this.%s", reservedName))
                .build();
    }

    private MethodSpec setter(String name, TypeName type) {

        String reservedName = reserved.contains(name) ? name + "_" : name;
        return MethodSpec.methodBuilder("set" + WordUtils.capitalize(reservedName))
                .addModifiers(Modifier.PUBLIC)
                .addParameter(type, reservedName)
                .addStatement(String.format("this.%s = %s", reservedName, reservedName))
                .build();
    }

    static Type regular(String type) {

        type = type.toLowerCase().trim();

        if (types.containsKey(type)) {
            return types.get(type);
        }

        return String.class;
    }

    private final static Map<String, Type> types = new ConcurrentHashMap<String, Type>() {{
        put("string", String.class);
        put("date", Date.class);
        put("int", Integer.class);
        put("number", Long.class);
        put("array", List.class);
        put("boolean", Boolean.class);
        put("objectid", ObjectId.class);
        put("object", Object.class);
    }};

    private final static List<Type> structureType = Arrays.asList(
            ModCommon.class,
            ModUser.class,
            ModGroup.class,
            ModFile.class,
            ModCategory.class
    );

    private final static List<String> lightReserved = Arrays.asList(
            "_id",
            "createAt",
            "updateAt",
            "createBy",
            "updateBy",
            "valid",
            "options"
    );

    public final static List<String> reserved = Arrays.asList(
            "abstract",
            "continue",
            "for",
            "new",
            "switch",
            "assert",
            "default",
            "goto",
            "package",
            "synchronized",
            "boolean",
            "do",
            "if",
            "private",
            "this",
            "break",
            "double",
            "implements",
            "protected",
            "throw",
            "byte",
            "else",
            "import",
            "public",
            "throws",
            "case",
            "enum",
            "instanceof",
            "return",
            "transient",
            "catch",
            "extends",
            "int",
            "short",
            "try",
            "char",
            "final",
            "interface",
            "static",
            "void",
            "class",
            "finally",
            "long",
            "strictfp",
            "volatile",
            "const",
            "float",
            "native",
            "super",
            "while"
    );

}
