package cloud.toolshed.jsup;

import static cloud.toolshed.jsup.jooq.generated.Tables.BOOLEAN_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.FLOAT_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.INTEGER_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.SET_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.STRING_PREFERENCES;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static org.jooq.impl.DSL.constraint;

import com.google.inject.AbstractModule;
import com.google.inject.BindingAnnotation;
import com.google.inject.Key;
import com.google.inject.Provides;
import org.jooq.DSLContext;
import org.jooq.SQLDialect;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.jooq.impl.SQLDataType;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.sql.DataSource;

/**
 * Binds an implementation of {@link UserPreferencesDb}.
 *
 * <p>Before installing this module a {@link DataSource} and {@link SQLDialect}
 * annotated with {@link UserPreferences} should be bound by another module.
 *
 * <p>For example:
 * <pre>
 * {@code
 * Injector injector = Guice.createInjector(
 *     new AbstractModule() {
 *       @Override
 *       protected void configure() {
 *         HikariConfig config = new HikariConfig();
 *         config.setJdbcUrl("jdbc:h2:mem:testdb");
 *         HikariDataSource dataSource = new HikariDataSource(config);
 *
 *         bind(Key.get(SQLDialect.class, UserPreferencesModule.UserPreferences.class)).toInstance(SQLDialect.H2);
 *         bind(Key.get(DataSource.class, UserPreferencesModule.UserPreferences.class)).toInstance(dataSource);
 *       }
 *     },
 *     new UserPreferencesModule());
 *
 * UserPreferencesDb userPreferencesDb = injector.getInstance(UserPreferencesDb.class);
 * UserPreferencesAnalytics userPreferencesAnalytics = injector.getInstance(UserPreferencesAnalytics.class);
 *
 * userPreferencesDb.setString("some-user", "ui", "theme", "default");
 * userPreferencesDb.setString("some-user", "ui", "display.timestamp.timezone", "UTC");
 * userPreferencesDb.addToSet("some-user", "app", "favorite.projects", ImmutableSet.of("prj1", "prj2"));
 * userPreferencesDb.setBoolean("some-user", "app", "email.on.failure", true);
 *
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "theme"));
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "display.timestamp.timezone"));
 * System.out.println(userPreferencesDb.getSet("some-user", "app", "favorite.projects"));
 * System.out.println(userPreferencesDb.getBoolean("some-user", "app", "email.on.failure"));
 *
 * System.out.println(userPreferencesAnalytics.getStringCategories());
 * System.out.println(userPreferencesAnalytics.getBooleanPreferencesByUser("app", "email.on.failure"));
 * }
 * </pre>
 *
 * @author Akshay Dayal
 */
public class UserPreferencesModule extends AbstractModule {

  @Override
  protected void configure() {
    requireBinding(Key.get(DataSource.class, UserPreferences.class));
    requireBinding(Key.get(SQLDialect.class, UserPreferences.class));
    bind(UserPreferencesDb.class).to(UserPreferencesDbImpl.class);
    bind(UserPreferencesAnalytics.class).to(UserPreferencesAnalyticsImpl.class);
  }

  /**
   * Provides the {@link DataSource} used by implementations of {@link UserPreferencesDb}.
   *
   * <p>When a {@link UserPreferencesDb} is requested for the first time this provider will establish a connection to
   * the SQL database using the {@link DataSource} annotated with {@link UserPreferences} and if needed user preference
   * tables are created. This removes the need for users to create tables manually beforehand.
   *
   * @param dataSource The {@link DataSource} annotated with {@link UserPreferences}.
   * @param sqlDialect The {@link SQLDialect} annotated with {@link UserPreferences}.
   * @return The same {@link DataSource} once tables have been created if needed.
   */
  @Inject
  @Provides
  @Singleton
  @UserPreferencesInternal
  DataSource provideDataSource(@UserPreferences DataSource dataSource, @UserPreferences SQLDialect sqlDialect) {
    DSLContext dslContext = DSL.using(dataSource, sqlDialect);

    try {
      createSchemaIfNotExists(dslContext, sqlDialect);

      dslContext.createTableIfNotExists(BOOLEAN_PREFERENCES)
          .column(BOOLEAN_PREFERENCES.USERNAME, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(BOOLEAN_PREFERENCES.CATEGORY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(BOOLEAN_PREFERENCES.KEY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(BOOLEAN_PREFERENCES.VALUE, SQLDataType.TINYINT.nullable(false))
          .constraints(
              constraint("PK_" + BOOLEAN_PREFERENCES.getName())
                  .primaryKey(BOOLEAN_PREFERENCES.USERNAME,
                      BOOLEAN_PREFERENCES.CATEGORY,
                      BOOLEAN_PREFERENCES.KEY))
          .execute();
      dslContext.createIndex("I_" + BOOLEAN_PREFERENCES.getName())
          .on(BOOLEAN_PREFERENCES,
              BOOLEAN_PREFERENCES.CATEGORY,
              BOOLEAN_PREFERENCES.KEY,
              BOOLEAN_PREFERENCES.VALUE);

      dslContext.createTableIfNotExists(INTEGER_PREFERENCES)
          .column(INTEGER_PREFERENCES.USERNAME, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(INTEGER_PREFERENCES.CATEGORY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(INTEGER_PREFERENCES.KEY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(INTEGER_PREFERENCES.VALUE, SQLDataType.INTEGER.nullable(false))
          .constraints(
              constraint("PK_" + INTEGER_PREFERENCES.getName())
                  .primaryKey(INTEGER_PREFERENCES.USERNAME,
                      INTEGER_PREFERENCES.CATEGORY,
                      INTEGER_PREFERENCES.KEY))
          .execute();
      dslContext.createIndex("I_" + INTEGER_PREFERENCES.getName())
          .on(INTEGER_PREFERENCES,
              INTEGER_PREFERENCES.CATEGORY,
              INTEGER_PREFERENCES.KEY,
              INTEGER_PREFERENCES.VALUE);

      dslContext.createTableIfNotExists(FLOAT_PREFERENCES)
          .column(FLOAT_PREFERENCES.USERNAME, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(FLOAT_PREFERENCES.CATEGORY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(FLOAT_PREFERENCES.KEY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(FLOAT_PREFERENCES.VALUE, SQLDataType.FLOAT.nullable(false))
          .constraints(
              constraint("PK_" + FLOAT_PREFERENCES.getName())
                  .primaryKey(FLOAT_PREFERENCES.USERNAME,
                      FLOAT_PREFERENCES.CATEGORY,
                      FLOAT_PREFERENCES.KEY))
          .execute();
      dslContext.createIndex("I_" + FLOAT_PREFERENCES.getName())
          .on(FLOAT_PREFERENCES,
              FLOAT_PREFERENCES.CATEGORY,
              FLOAT_PREFERENCES.KEY,
              FLOAT_PREFERENCES.VALUE);

      dslContext.createTableIfNotExists(STRING_PREFERENCES)
          .column(STRING_PREFERENCES.USERNAME, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(STRING_PREFERENCES.CATEGORY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(STRING_PREFERENCES.KEY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(STRING_PREFERENCES.VALUE, SQLDataType.VARCHAR.length(512).nullable(false))
          .constraints(
              constraint("PK_" + STRING_PREFERENCES.getName())
                  .primaryKey(STRING_PREFERENCES.USERNAME,
                      STRING_PREFERENCES.CATEGORY,
                      STRING_PREFERENCES.KEY))
          .execute();
      dslContext.createIndex("I_" + STRING_PREFERENCES.getName())
          .on(STRING_PREFERENCES,
              STRING_PREFERENCES.CATEGORY,
              STRING_PREFERENCES.KEY,
              STRING_PREFERENCES.VALUE);

      dslContext.createTableIfNotExists(SET_PREFERENCES)
          .column(SET_PREFERENCES.USERNAME, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(SET_PREFERENCES.CATEGORY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(SET_PREFERENCES.KEY, SQLDataType.VARCHAR.length(512).nullable(false))
          .column(SET_PREFERENCES.VALUE, SQLDataType.VARCHAR.length(512).nullable(false))
          .constraints(
              constraint("PK_" + SET_PREFERENCES.getName())
                  .primaryKey(SET_PREFERENCES.USERNAME,
                      SET_PREFERENCES.CATEGORY,
                      SET_PREFERENCES.KEY,
                      SET_PREFERENCES.VALUE))
          .execute();
      dslContext.createIndex("I_" + SET_PREFERENCES.getName())
          .on(SET_PREFERENCES,
              SET_PREFERENCES.CATEGORY,
              SET_PREFERENCES.KEY,
              SET_PREFERENCES.VALUE);
    } catch (DataAccessException dae) {
      throw new RuntimeException("Unable to create tables", dae);
    }

    return dataSource;
  }

  private void createSchemaIfNotExists(DSLContext dslContext, SQLDialect sqlDialect) {
    String schemaName = "jsup";
    if (sqlDialect.equals(SQLDialect.H2) || sqlDialect.equals(SQLDialect.HSQLDB)) {
      schemaName = "\"jsup\"";
    }

    if (sqlDialect.equals(SQLDialect.HSQLDB)) {
      try {
        dslContext.execute("create schema " + schemaName);
      } catch (Exception ex) {
        if (!ex.getMessage().contains("already exists")) {
          throw ex;
        }
      }
    } else {
      dslContext.execute("create schema if not exists " + schemaName);
    }
  }

  /**
   * A {@link DataSource} and {@link SQLDialect} with this annotation must be bound beforehand.
   */
  @BindingAnnotation
  @Target({ FIELD, PARAMETER, METHOD })
  @Retention(RUNTIME)
  public @interface UserPreferences {}

  /**
   * For internal use.
   */
  @BindingAnnotation
  @Target({ FIELD, PARAMETER, METHOD })
  @Retention(RUNTIME)
  @interface UserPreferencesInternal {}
}
