package cloud.toolshed.jsup;

import static cloud.toolshed.jsup.jooq.generated.Tables.BOOLEAN_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.FLOAT_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.INTEGER_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.SET_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.STRING_PREFERENCES;
import static org.jooq.impl.DSL.field;

import com.google.common.collect.ImmutableSet;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SQLDialect;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;

import java.io.IOException;
import java.util.Set;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.sql.DataSource;

/**
 * SQL based implementation of {@link UserPreferencesDb}.
 *
 * <p>Uses jOOQ for compatibility with a variety of SQL databases. A passed in
 * {@link SQLDialect} will let this implementation know which exact SQL
 * database is being used and that is passed along to jOOQ for making sure the
 * correct SQL queries are generated.
 *
 * <p>Refer to the jOOQ documentation for a complete list of supported SQL
 * databases.
 *
 * <p>For example:
 * <pre>
 * {@code
 * Injector injector = Guice.createInjector(
 *     new AbstractModule() {
 *       @Override
 *       protected void configure() {
 *         HikariConfig config = new HikariConfig();
 *         config.setJdbcUrl("jdbc:h2:mem:testdb");
 *         HikariDataSource dataSource = new HikariDataSource(config);
 *
 *         bind(Key.get(SQLDialect.class, UserPreferencesModule.UserPreferences.class)).toInstance(SQLDialect.H2);
 *         bind(Key.get(DataSource.class, UserPreferencesModule.UserPreferences.class)).toInstance(dataSource);
 *       }
 *     },
 *     new UserPreferencesModule());
 *
 * UserPreferencesDb userPreferencesDb = injector.getInstance(UserPreferencesDb.class);
 *
 * userPreferencesDb.setString("some-user", "ui", "theme", "default");
 * userPreferencesDb.setString("some-user", "ui", "display.timestamp.timezone", "UTC");
 * userPreferencesDb.addToSet("some-user", "app", "favorite.projects", ImmutableSet.of("prj1", "prj2"));
 * userPreferencesDb.setBoolean("some-user", "app", "email.on.failure", true);
 *
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "theme"));
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "display.timestamp.timezone"));
 * System.out.println(userPreferencesDb.getSet("some-user", "app", "favorite.projects"));
 * System.out.println(userPreferencesDb.getBoolean("some-user", "app", "email.on.failure"));
 * }
 * </pre>
 *
 * @author Akshay Dayal
 */
@Singleton
class UserPreferencesDbImpl implements UserPreferencesDb {

  private final DataSource dataSource;
  private final SQLDialect sqlDialect;

  @Inject
  UserPreferencesDbImpl(
      @UserPreferencesModule.UserPreferencesInternal DataSource dataSource,
      @UserPreferencesModule.UserPreferences SQLDialect sqlDialect) {
    this.dataSource = dataSource;
    this.sqlDialect = sqlDialect;
  }

  @Nullable
  @Override
  public Boolean getBoolean(String username, String category, String preferenceName) throws IOException {
    try {
      Result<Record> result = createDslContext().select()
          .from(BOOLEAN_PREFERENCES)
          .where(BOOLEAN_PREFERENCES.USERNAME.eq(username))
          .and(BOOLEAN_PREFERENCES.CATEGORY.eq(category))
          .and(BOOLEAN_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return null;
      }

      return result.get(0).get(BOOLEAN_PREFERENCES.VALUE) != 0;
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to fetch from %s %s/%s/%s",
              BOOLEAN_PREFERENCES.getName(), username, category, preferenceName), dae);
    }
  }

  @Override
  public void setBoolean(
      String username,
      String category,
      String preferenceName,
      @Nullable Boolean value) throws IOException {
    try {
      if (value == null) {
        // Delete the existing preference.
        createDslContext().delete(BOOLEAN_PREFERENCES)
            .where(BOOLEAN_PREFERENCES.USERNAME.eq(username))
            .and(BOOLEAN_PREFERENCES.CATEGORY.eq(category))
            .and(BOOLEAN_PREFERENCES.KEY.eq(preferenceName))
            .execute();
      } else {
        // Insert or update the preference.
        byte byteValue = !value ? (byte) 0 : (byte) 1;
        createDslContext()
            .mergeInto(
                BOOLEAN_PREFERENCES,
                field(BOOLEAN_PREFERENCES.USERNAME),
                field(BOOLEAN_PREFERENCES.CATEGORY),
                field(BOOLEAN_PREFERENCES.KEY),
                field(BOOLEAN_PREFERENCES.VALUE))
            .key(
                field(BOOLEAN_PREFERENCES.USERNAME),
                field(BOOLEAN_PREFERENCES.CATEGORY),
                field(BOOLEAN_PREFERENCES.KEY))
            .values(username, category, preferenceName, byteValue)
            .execute();
      }
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to save into %s %s/%s/%s: %s",
              BOOLEAN_PREFERENCES.getName(), username, category, preferenceName, value), dae);
    }
  }

  @Nullable
  @Override
  public Integer getInteger(String username, String category, String preferenceName) throws IOException {
    try {
      Result<Record> result = createDslContext().select()
          .from(INTEGER_PREFERENCES)
          .where(INTEGER_PREFERENCES.USERNAME.eq(username))
          .and(INTEGER_PREFERENCES.CATEGORY.eq(category))
          .and(INTEGER_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return null;
      }

      return result.get(0).get(INTEGER_PREFERENCES.VALUE);
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to fetch from %s %s/%s/%s",
              INTEGER_PREFERENCES.getName(), username, category, preferenceName), dae);
    }
  }

  @Override
  public void setInteger(
      String username,
      String category,
      String preferenceName,
      @Nullable Integer value) throws IOException {
    try {
      if (value == null) {
        // Delete the existing preference.
        createDslContext().delete(INTEGER_PREFERENCES)
            .where(INTEGER_PREFERENCES.USERNAME.eq(username))
            .and(INTEGER_PREFERENCES.CATEGORY.eq(category))
            .and(INTEGER_PREFERENCES.KEY.eq(preferenceName))
            .execute();
      } else {
        // Insert or update the preference.
        createDslContext()
            .mergeInto(
                INTEGER_PREFERENCES,
                field(INTEGER_PREFERENCES.USERNAME),
                field(INTEGER_PREFERENCES.CATEGORY),
                field(INTEGER_PREFERENCES.KEY),
                field(INTEGER_PREFERENCES.VALUE))
            .key(
                field(INTEGER_PREFERENCES.USERNAME),
                field(INTEGER_PREFERENCES.CATEGORY),
                field(INTEGER_PREFERENCES.KEY))
            .values(username, category, preferenceName, value)
            .execute();
      }
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to save into %s %s/%s/%s: %s",
              INTEGER_PREFERENCES.getName(), username, category, preferenceName, value), dae);
    }
  }

  @Nullable
  @Override
  public Float getFloat(String username, String category, String preferenceName) throws IOException {
    try {
      Result<Record> result = createDslContext().select()
          .from(FLOAT_PREFERENCES)
          .where(FLOAT_PREFERENCES.USERNAME.eq(username))
          .and(FLOAT_PREFERENCES.CATEGORY.eq(category))
          .and(FLOAT_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return null;
      }

      return (float) (double) result.get(0).get(FLOAT_PREFERENCES.VALUE);
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to fetch from %s %s/%s/%s",
              FLOAT_PREFERENCES.getName(), username, category, preferenceName), dae);
    }
  }

  @Override
  public void setFloat(
      String username,
      String category,
      String preferenceName,
      @Nullable Float value) throws IOException {
    try {
      if (value == null) {
        // Delete the existing preference.
        createDslContext().delete(FLOAT_PREFERENCES)
            .where(FLOAT_PREFERENCES.USERNAME.eq(username))
            .and(FLOAT_PREFERENCES.CATEGORY.eq(category))
            .and(FLOAT_PREFERENCES.KEY.eq(preferenceName))
            .execute();
      } else {
        // Insert or update the preference.
        createDslContext()
            .mergeInto(
                FLOAT_PREFERENCES,
                field(FLOAT_PREFERENCES.USERNAME),
                field(FLOAT_PREFERENCES.CATEGORY),
                field(FLOAT_PREFERENCES.KEY),
                field(FLOAT_PREFERENCES.VALUE))
            .key(
                field(FLOAT_PREFERENCES.USERNAME),
                field(FLOAT_PREFERENCES.CATEGORY),
                field(FLOAT_PREFERENCES.KEY))
            .values(username, category, preferenceName, (double) value)
            .execute();
      }
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to save into %s %s/%s/%s: %s",
              FLOAT_PREFERENCES.getName(), username, category, preferenceName, value), dae);
    }
  }

  @Nullable
  @Override
  public String getString(String username, String category, String preferenceName) throws IOException {
    try {
      Result<Record> result = createDslContext().select()
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.USERNAME.eq(username))
          .and(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return null;
      }

      return result.get(0).get(STRING_PREFERENCES.VALUE);
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to fetch from %s %s/%s/%s",
              STRING_PREFERENCES.getName(), username, category, preferenceName), dae);
    }
  }

  @Override
  public void setString(
      String username,
      String category,
      String preferenceName,
      @Nullable String value) throws IOException {
    try {
      if (value == null) {
        // Delete the existing preference.
        createDslContext().delete(STRING_PREFERENCES)
            .where(STRING_PREFERENCES.USERNAME.eq(username))
            .and(STRING_PREFERENCES.CATEGORY.eq(category))
            .and(STRING_PREFERENCES.KEY.eq(preferenceName))
            .execute();
      } else {
        // Insert or update the preference.
        createDslContext()
            .mergeInto(
                STRING_PREFERENCES,
                field(STRING_PREFERENCES.USERNAME),
                field(STRING_PREFERENCES.CATEGORY),
                field(STRING_PREFERENCES.KEY),
                field(STRING_PREFERENCES.VALUE))
            .key(
                field(STRING_PREFERENCES.USERNAME),
                field(STRING_PREFERENCES.CATEGORY),
                field(STRING_PREFERENCES.KEY))
            .values(username, category, preferenceName, value)
            .execute();
      }
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to save into %s %s/%s/%s: %s",
              STRING_PREFERENCES.getName(), username, category, preferenceName, value), dae);
    }
  }

  @Override
  public ImmutableSet<String> getSet(String username, String category, String preferenceName) throws IOException {
    try {
      Result<Record> result = createDslContext().select()
          .from(SET_PREFERENCES)
          .where(SET_PREFERENCES.USERNAME.eq(username))
          .and(SET_PREFERENCES.CATEGORY.eq(category))
          .and(SET_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record record : result) {
        builder.add(record.get(SET_PREFERENCES.VALUE));
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(
          String.format("Failed to fetch from %s %s/%s/%s",
              SET_PREFERENCES.getName(), username, category, preferenceName), dae);
    }
  }

  @Override
  public void addToSet(
      String username,
      String category,
      String preferenceName,
      Set<String> values) throws IOException {
    DSLContext dslContext = createDslContext();
    for (String value : values) {
      try {
        if (sqlDialect.equals(SQLDialect.HSQLDB)) {
          dslContext.insertInto(
              SET_PREFERENCES,
              SET_PREFERENCES.USERNAME,
              SET_PREFERENCES.CATEGORY,
              SET_PREFERENCES.KEY,
              SET_PREFERENCES.VALUE)
              .values(username, category, preferenceName, value)
              .onDuplicateKeyIgnore()
              .execute();
        } else {
          dslContext
              .mergeInto(
                  SET_PREFERENCES,
                  field(SET_PREFERENCES.USERNAME),
                  field(SET_PREFERENCES.CATEGORY),
                  field(SET_PREFERENCES.KEY),
                  field(SET_PREFERENCES.VALUE))
              .key(
                  field(SET_PREFERENCES.USERNAME),
                  field(SET_PREFERENCES.CATEGORY),
                  field(SET_PREFERENCES.KEY),
                  field(SET_PREFERENCES.VALUE))
              .values(username, category, preferenceName, value)
              .execute();
        }
      } catch (DataAccessException dae) {
        throw new IOException(String.format(
            "Failed to add into %s %s/%s/%s: %s",
            SET_PREFERENCES.getName(), username, category, preferenceName, value), dae);
      }
    }
  }

  @Override
  public void removeFromSet(
      String username,
      String category,
      String preferenceName,
      @Nullable Set<String> values) throws IOException {
    DSLContext dslContext = createDslContext();

    if (values == null) {
      // Delete the entire preference.
      try {
        dslContext
            .delete(SET_PREFERENCES)
            .where(SET_PREFERENCES.USERNAME.eq(username))
            .and(SET_PREFERENCES.CATEGORY.eq(category))
            .and(SET_PREFERENCES.KEY.eq(preferenceName))
            .execute();
        return;
      } catch (DataAccessException dae) {
        throw new IOException(String.format(
            "Failed to remove all %s %s/%s/%s",
            SET_PREFERENCES.getName(), username, category, preferenceName), dae);
      }
    }

    // Delete only the specified values.
    for (String value : values) {
      try {
        dslContext
            .delete(SET_PREFERENCES)
            .where(SET_PREFERENCES.USERNAME.eq(username))
            .and(SET_PREFERENCES.CATEGORY.eq(category))
            .and(SET_PREFERENCES.KEY.eq(preferenceName))
            .and(SET_PREFERENCES.VALUE.eq(value))
            .execute();
      } catch (DataAccessException dae) {
        throw new IOException(String.format(
            "Failed to remove from %s %s/%s/%s: %s",
            SET_PREFERENCES.getName(), username, category, preferenceName, value), dae);
      }
    }
  }

  private DSLContext createDslContext() {
    return DSL.using(dataSource, sqlDialect);
  }
}
