package cloud.toolshed.jsup;

import com.google.common.collect.ImmutableSet;

import java.io.IOException;
import java.util.Set;
import javax.annotation.Nullable;

/**
 * Simple database to store and fetch user preferences by category.
 *
 * <p>Supports preferences of the following types:
 * <ul>
 *   <li>{@link Boolean}</li>
 *   <li>{@link Integer}</li>
 *   <li>{@link Float}</li>
 *   <li>{@link String}</li>
 *   <li>{@link Set} of Strings</li>
 * </ul>
 *
 * <p>Strings can be a maximum size of 512 characters.
 *
 * <p>For example:
 * <pre>
 * {@code
 * Injector injector = Guice.createInjector(
 *     new AbstractModule() {
 *       @Override
 *       protected void configure() {
 *         HikariConfig config = new HikariConfig();
 *         config.setJdbcUrl("jdbc:h2:mem:testdb");
 *         HikariDataSource dataSource = new HikariDataSource(config);
 *
 *         bind(Key.get(SQLDialect.class, UserPreferencesModule.UserPreferences.class)).toInstance(SQLDialect.H2);
 *         bind(Key.get(DataSource.class, UserPreferencesModule.UserPreferences.class)).toInstance(dataSource);
 *       }
 *     },
 *     new UserPreferencesModule());
 *
 * UserPreferencesDb userPreferencesDb = injector.getInstance(UserPreferencesDb.class);
 *
 * userPreferencesDb.setString("some-user", "ui", "theme", "default");
 * userPreferencesDb.setString("some-user", "ui", "display.timestamp.timezone", "UTC");
 * userPreferencesDb.addToSet("some-user", "app", "favorite.projects", ImmutableSet.of("prj1", "prj2"));
 * userPreferencesDb.setBoolean("some-user", "app", "email.on.failure", true);
 *
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "theme"));
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "display.timestamp.timezone"));
 * System.out.println(userPreferencesDb.getSet("some-user", "app", "favorite.projects"));
 * System.out.println(userPreferencesDb.getBoolean("some-user", "app", "email.on.failure"));
 * }
 * </pre>
 *
 * @author Akshay Dayal
 */
public interface UserPreferencesDb {

  /**
   * Get the value of a boolean preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   *
   * @return The value if defined for the preference, else null.
   *
   * @throws IOException If there is an error executing the query.
   */
  @Nullable
  Boolean getBoolean(String username, String category, String preferenceName) throws IOException;

  /**
   * Set the value of a boolean preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param value The value to set for the preference. If null then the preference is deleted for the user.
   *
   * @throws IOException If there is an error saving the preference.
   */
  void setBoolean(String username, String category, String preferenceName, @Nullable Boolean value) throws IOException;

  /**
   * Get the value of an integer preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   *
   * @return The value if defined for the preference, else null.
   *
   * @throws IOException If there is an error executing the query.
   */
  @Nullable
  Integer getInteger(String username, String category, String preferenceName) throws IOException;

  /**
   * Set the value of an integer preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param value The value to set for the preference. If null then the preference is deleted for the user.
   *
   * @throws IOException If there is an error saving the preference.
   */
  void setInteger(String username, String category, String preferenceName, @Nullable Integer value) throws IOException;

  /**
   * Get the value of a float preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   *
   * @return The value if defined for the preference, else null.
   *
   * @throws IOException If there is an error saving the preference.
   */
  @Nullable
  Float getFloat(String username, String category, String preferenceName) throws IOException;

  /**
   * Set the value of a float preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param value The value to set for the preference. If null then the preference is deleted for the user.
   *
   * @throws IOException If there is an error saving the preference.
   */
  void setFloat(String username, String category, String preferenceName, @Nullable Float value) throws IOException;

  /**
   * Get the value of a string preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   *
   * @return The value if defined for the preference, else null.
   *
   * @throws IOException If there is an error executing the query.
   */
  @Nullable
  String getString(String username, String category, String preferenceName) throws IOException;

  /**
   * Set the value of a string preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param value The value to set for the preference. If null then the preference is deleted for the user.
   *
   * @throws IOException If there is an error saving the preference.
   */
  void setString(String username, String category, String preferenceName, @Nullable String value) throws IOException;

  /**
   * Get the values of a {@link Set} preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   *
   * @return The values if defined for the preference, else an empty set.
   *
   * @throws IOException If there is an error executing the query.
   */
  ImmutableSet<String> getSet(String username, String category, String preferenceName) throws IOException;

  /**
   * Add values to a {@link Set} preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param values The values to add.
   *
   * @throws IOException If there is an error updating the preference.
   */
  void addToSet(String username, String category, String preferenceName, Set<String> values) throws IOException;

  /**
   * Remove values from a {@link Set} preference.
   *
   * @param username The username whose preference to look for.
   * @param category The preference category.
   * @param preferenceName The name of the preference.
   * @param values The values to remove. If null then the entire preference is deleted.
   *
   * @throws IOException If there is an error updating the preference.
   */
  void removeFromSet(
      String username, String category, String preferenceName, @Nullable Set<String> values) throws IOException;
}
