package cloud.toolshed.jsup;

import static cloud.toolshed.jsup.jooq.generated.Tables.BOOLEAN_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.FLOAT_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.INTEGER_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.SET_PREFERENCES;
import static cloud.toolshed.jsup.jooq.generated.Tables.STRING_PREFERENCES;

import com.google.common.collect.BoundType;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Range;
import org.jooq.DSLContext;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.SQLDialect;
import org.jooq.SelectConditionStep;
import org.jooq.TableField;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.jooq.impl.TableImpl;

import java.io.IOException;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.sql.DataSource;

/**
 * Implementation of {@link UserPreferencesAnalytics} that takes advantage of
 * secondary indexes to provide more efficient queries than doing scans and
 * filtering.
 *
 * <p>For example:
 * <pre>
 * {@code
 * Injector injector = Guice.createInjector(
 *     new AbstractModule() {
 *       @Override
 *       protected void configure() {
 *         HikariConfig config = new HikariConfig();
 *         config.setJdbcUrl("jdbc:h2:mem:testdb");
 *         HikariDataSource dataSource = new HikariDataSource(config);
 *
 *         bind(Key.get(SQLDialect.class, UserPreferencesModule.UserPreferences.class)).toInstance(SQLDialect.H2);
 *         bind(Key.get(DataSource.class, UserPreferencesModule.UserPreferences.class)).toInstance(dataSource);
 *       }
 *     },
 *     new UserPreferencesModule());
 *
 * UserPreferencesDb userPreferencesDb = injector.getInstance(UserPreferencesDb.class);
 * UserPreferencesAnalytics userPreferencesAnalytics = injector.getInstance(UserPreferencesAnalytics.class);
 *
 * userPreferencesDb.setString("some-user", "ui", "theme", "default");
 * userPreferencesDb.setString("some-user", "ui", "display.timestamp.timezone", "UTC");
 * userPreferencesDb.addToSet("some-user", "app", "favorite.projects", ImmutableSet.of("prj1", "prj2"));
 * userPreferencesDb.setBoolean("some-user", "app", "email.on.failure", true);
 *
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "theme"));
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "display.timestamp.timezone"));
 * System.out.println(userPreferencesDb.getSet("some-user", "app", "favorite.projects"));
 * System.out.println(userPreferencesDb.getBoolean("some-user", "app", "email.on.failure"));
 *
 * System.out.println(userPreferencesAnalytics.getStringCategories());
 * System.out.println(userPreferencesAnalytics.getBooleanPreferencesByUser("app", "email.on.failure"));
 * }
 * </pre>
 *
 * @author Akshay Dayal
 */
@Singleton
class UserPreferencesAnalyticsImpl implements UserPreferencesAnalytics {

  private final DataSource dataSource;
  private final SQLDialect sqlDialect;

  @Inject
  UserPreferencesAnalyticsImpl(
      @UserPreferencesModule.UserPreferencesInternal DataSource dataSource,
      @UserPreferencesModule.UserPreferences SQLDialect sqlDialect) {
    this.dataSource = dataSource;
    this.sqlDialect = sqlDialect;
  }

  @Override
  public ImmutableSet<String> getBooleanCategories() throws IOException {
    return getCategories(BOOLEAN_PREFERENCES, BOOLEAN_PREFERENCES.CATEGORY);
  }

  @Override
  public ImmutableSet<String> getIntegerCategories() throws IOException {
    return getCategories(INTEGER_PREFERENCES, INTEGER_PREFERENCES.CATEGORY);
  }

  @Override
  public ImmutableSet<String> getFloatCategories() throws IOException {
    return getCategories(FLOAT_PREFERENCES, FLOAT_PREFERENCES.CATEGORY);
  }

  @Override
  public ImmutableSet<String> getStringCategories() throws IOException {
    return getCategories(STRING_PREFERENCES, STRING_PREFERENCES.CATEGORY);
  }

  @Override
  public ImmutableSet<String> getSetCategories() throws IOException {
    return getCategories(SET_PREFERENCES, SET_PREFERENCES.CATEGORY);
  }

  private ImmutableSet<String> getCategories(
      TableImpl<?> table, TableField<?, String> categoryField) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(categoryField)
          .from(table)
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException("Failed to fetch categories from: " + table.getName(), dae);
    }
  }

  @Override
  public ImmutableSet<String> getBooleanPreferenceNames(String category) throws IOException {
    return getPreferenceNames(BOOLEAN_PREFERENCES, BOOLEAN_PREFERENCES.CATEGORY, category, BOOLEAN_PREFERENCES.KEY);
  }

  @Override
  public ImmutableSet<String> getIntegerPreferenceNames(String category) throws IOException {
    return getPreferenceNames(INTEGER_PREFERENCES, INTEGER_PREFERENCES.CATEGORY, category, INTEGER_PREFERENCES.KEY);
  }

  @Override
  public ImmutableSet<String> getFloatPreferenceNames(String category) throws IOException {
    return getPreferenceNames(FLOAT_PREFERENCES, FLOAT_PREFERENCES.CATEGORY, category, FLOAT_PREFERENCES.KEY);
  }

  @Override
  public ImmutableSet<String> getStringPreferenceNames(String category) throws IOException {
    return getPreferenceNames(STRING_PREFERENCES, STRING_PREFERENCES.CATEGORY, category, STRING_PREFERENCES.KEY);
  }

  @Override
  public ImmutableSet<String> getSetPreferenceNames(String category) throws IOException {
    return getPreferenceNames(SET_PREFERENCES, SET_PREFERENCES.CATEGORY, category, SET_PREFERENCES.KEY);
  }

  private ImmutableSet<String> getPreferenceNames(
      TableImpl<?> table,
      TableField<?, String> categoryField,
      String categoryValue,
      TableField<?, String> keyField) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(keyField)
          .from(table)
          .where(categoryField.eq(categoryValue))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences from %s for category %s", table.getName(), categoryValue), dae);
    }
  }

  @Override
  public ImmutableMap<String, Boolean> getBooleanPreferencesByUser(
      String category, String preferenceName) throws IOException {
    try {
      Result<Record2<String, Byte>> result = createDslContext()
          .select(BOOLEAN_PREFERENCES.USERNAME, BOOLEAN_PREFERENCES.VALUE)
          .from(BOOLEAN_PREFERENCES)
          .where(BOOLEAN_PREFERENCES.CATEGORY.eq(category))
          .and(BOOLEAN_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, Boolean> builder = ImmutableMap.builder();
      for (Record2<String, Byte> record : result) {
        builder.put(record.value1(), record.value2() != 0);
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s",
          BOOLEAN_PREFERENCES.getName(), category, preferenceName), dae);
    }
  }

  @Override
  public ImmutableSet<String> getUsersWithBooleanPreference(
      String category, String preferenceName, boolean value) throws IOException {
    try {
      byte byteValue = !value ? (byte) 0 : (byte) 1;
      Result<Record1<String>> result = createDslContext().select(BOOLEAN_PREFERENCES.USERNAME)
          .from(BOOLEAN_PREFERENCES)
          .where(BOOLEAN_PREFERENCES.CATEGORY.eq(category))
          .and(BOOLEAN_PREFERENCES.KEY.eq(preferenceName))
          .and(BOOLEAN_PREFERENCES.VALUE.eq(byteValue))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch users from %s for %s/%s with value %s",
          BOOLEAN_PREFERENCES.getName(), category, preferenceName, value), dae);
    }
  }

  @Override
  public ImmutableMap<String, Integer> getIntegerPreferencesByUser(
      String category, String preferenceName) throws IOException {
    try {
      Result<Record2<String, Integer>> result = createDslContext()
          .select(INTEGER_PREFERENCES.USERNAME, INTEGER_PREFERENCES.VALUE)
          .from(INTEGER_PREFERENCES)
          .where(INTEGER_PREFERENCES.CATEGORY.eq(category))
          .and(INTEGER_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, Integer> builder = ImmutableMap.builder();
      for (Record2<String, Integer> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s",
          INTEGER_PREFERENCES.getName(), category, preferenceName), dae);
    }
  }

  @Override
  public ImmutableMap<String, Integer> getIntegerPreferencesByUser(
      String category, String preferenceName, Range<Integer> range) throws IOException {
    if (!range.hasLowerBound() && !range.hasUpperBound()) {
      return getIntegerPreferencesByUser(category, preferenceName);
    }

    try {
      SelectConditionStep<Record2<String, Integer>> query = createDslContext()
          .select(INTEGER_PREFERENCES.USERNAME, INTEGER_PREFERENCES.VALUE)
          .from(INTEGER_PREFERENCES)
          .where(INTEGER_PREFERENCES.CATEGORY.eq(category))
          .and(INTEGER_PREFERENCES.KEY.eq(preferenceName));

      if (range.hasLowerBound()) {
        if (range.lowerBoundType().equals(BoundType.CLOSED)) {
          query = query.and(INTEGER_PREFERENCES.VALUE.ge(range.lowerEndpoint()));
        } else {
          query = query.and(INTEGER_PREFERENCES.VALUE.gt(range.lowerEndpoint()));
        }
      }
      if (range.hasUpperBound()) {
        if (range.upperBoundType().equals(BoundType.CLOSED)) {
          query = query.and(INTEGER_PREFERENCES.VALUE.le(range.upperEndpoint()));
        } else {
          query = query.and(INTEGER_PREFERENCES.VALUE.lt(range.upperEndpoint()));
        }
      }

      Result<Record2<String, Integer>> result = query.fetch();
      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, Integer> builder = ImmutableMap.builder();
      for (Record2<String, Integer> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s and range %s",
          INTEGER_PREFERENCES.getName(), category, preferenceName, range.toString()), dae);
    }
  }

  @Override
  public ImmutableSet<String> getUsersWithIntegerPreference(
      String category, String preferenceName, int value) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(INTEGER_PREFERENCES.USERNAME)
          .from(INTEGER_PREFERENCES)
          .where(INTEGER_PREFERENCES.CATEGORY.eq(category))
          .and(INTEGER_PREFERENCES.KEY.eq(preferenceName))
          .and(INTEGER_PREFERENCES.VALUE.eq(value))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch users from %s for %s/%s with value %s",
          INTEGER_PREFERENCES.getName(), category, preferenceName, value), dae);
    }
  }

  @Override
  public ImmutableMap<String, Float> getFloatPreferencesByUser(
      String category, String preferenceName) throws IOException {
    try {
      Result<Record2<String, Double>> result = createDslContext()
          .select(FLOAT_PREFERENCES.USERNAME, FLOAT_PREFERENCES.VALUE)
          .from(FLOAT_PREFERENCES)
          .where(FLOAT_PREFERENCES.CATEGORY.eq(category))
          .and(FLOAT_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String,Float> builder = ImmutableMap.builder();
      for (Record2<String, Double> record : result) {
        builder.put(record.value1(), (float) (double) record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s",
          FLOAT_PREFERENCES.getName(), category, preferenceName), dae);
    }
  }

  @Override
  public ImmutableMap<String, Float> getFloatPreferencesByUser(
      String category, String preferenceName, Range<Float> range) throws IOException {
    if (!range.hasLowerBound() && !range.hasUpperBound()) {
      return getFloatPreferencesByUser(category, preferenceName);
    }

    try {
      SelectConditionStep<Record2<String, Double>> query = createDslContext()
          .select(FLOAT_PREFERENCES.USERNAME, FLOAT_PREFERENCES.VALUE)
          .from(FLOAT_PREFERENCES)
          .where(FLOAT_PREFERENCES.CATEGORY.eq(category))
          .and(FLOAT_PREFERENCES.KEY.eq(preferenceName));

      if (range.hasLowerBound()) {
        if (range.lowerBoundType().equals(BoundType.CLOSED)) {
          query = query.and(FLOAT_PREFERENCES.VALUE.ge((double) range.lowerEndpoint()));
        } else {
          query = query.and(FLOAT_PREFERENCES.VALUE.gt((double) range.lowerEndpoint()));
        }
      }
      if (range.hasUpperBound()) {
        if (range.upperBoundType().equals(BoundType.CLOSED)) {
          query = query.and(FLOAT_PREFERENCES.VALUE.le((double) range.upperEndpoint()));
        } else {
          query = query.and(FLOAT_PREFERENCES.VALUE.lt((double) range.upperEndpoint()));
        }
      }

      Result<Record2<String, Double>> result = query.fetch();
      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, Float> builder = ImmutableMap.builder();
      for (Record2<String, Double> record : result) {
        builder.put(record.value1(), (float) (double) record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s and range %s",
          FLOAT_PREFERENCES.getName(), category, preferenceName, range.toString()), dae);
    }
  }

  @Override
  public ImmutableSet<String> getUsersWithFloatPreference(
      String category, String preferenceName, float value) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(FLOAT_PREFERENCES.USERNAME)
          .from(FLOAT_PREFERENCES)
          .where(FLOAT_PREFERENCES.CATEGORY.eq(category))
          .and(FLOAT_PREFERENCES.KEY.eq(preferenceName))
          .and(FLOAT_PREFERENCES.VALUE.eq((double) value))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch users from %s for %s/%s with value %s",
          FLOAT_PREFERENCES.getName(), category, preferenceName, value), dae);
    }
  }

  @Override
  public ImmutableMap<String, String> getStringPreferencesByUser(
      String category, String preferenceName) throws IOException {
    try {
      Result<Record2<String, String>> result = createDslContext()
          .select(STRING_PREFERENCES.USERNAME, STRING_PREFERENCES.VALUE)
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
      for (Record2<String, String> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s",
          STRING_PREFERENCES.getName(), category, preferenceName), dae);
    }
  }

  @Override
  public ImmutableMap<String, String> getStringPreferencesByUserStartingWith(
      String category, String preferenceName, String startsWith) throws IOException {
    try {
      Result<Record2<String, String>> result = createDslContext()
          .select(STRING_PREFERENCES.USERNAME, STRING_PREFERENCES.VALUE)
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .and(STRING_PREFERENCES.VALUE.startsWith(startsWith))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
      for (Record2<String, String> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s starting with %s",
          STRING_PREFERENCES.getName(), category, preferenceName, startsWith), dae);
    }
  }

  @Override
  public ImmutableMap<String, String> getStringPreferencesByUserEndingWith(
      String category, String preferenceName, String endsWith) throws IOException {
    try {
      Result<Record2<String, String>> result = createDslContext()
          .select(STRING_PREFERENCES.USERNAME, STRING_PREFERENCES.VALUE)
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .and(STRING_PREFERENCES.VALUE.endsWith(endsWith))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
      for (Record2<String, String> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s ending with %s",
          STRING_PREFERENCES.getName(), category, preferenceName, endsWith), dae);
    }
  }

  @Override
  public ImmutableMap<String, String> getStringPreferencesByUserContaining(
      String category, String preferenceName, String containing) throws IOException {
    try {
      Result<Record2<String, String>> result = createDslContext()
          .select(STRING_PREFERENCES.USERNAME, STRING_PREFERENCES.VALUE)
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .and(STRING_PREFERENCES.VALUE.contains(containing))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableMap.of();
      }

      ImmutableMap.Builder<String, String> builder = ImmutableMap.builder();
      for (Record2<String, String> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s containing %s",
          STRING_PREFERENCES.getName(), category, preferenceName, containing), dae);
    }
  }

  @Override
  public ImmutableSet<String> getUsersWithStringPreference(
      String category, String preferenceName, String value) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(STRING_PREFERENCES.USERNAME)
          .from(STRING_PREFERENCES)
          .where(STRING_PREFERENCES.CATEGORY.eq(category))
          .and(STRING_PREFERENCES.KEY.eq(preferenceName))
          .and(STRING_PREFERENCES.VALUE.eq(value))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch users from %s for %s/%s with value %s",
          STRING_PREFERENCES.getName(), category, preferenceName, value), dae);
    }
  }

  @Override
  public ImmutableSetMultimap<String, String> getSetPreferencesByUser(
      String category, String preferenceName) throws IOException {
    try {
      Result<Record2<String, String>> result = createDslContext()
          .select(SET_PREFERENCES.USERNAME, SET_PREFERENCES.VALUE)
          .from(SET_PREFERENCES)
          .where(SET_PREFERENCES.CATEGORY.eq(category))
          .and(SET_PREFERENCES.KEY.eq(preferenceName))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSetMultimap.of();
      }

      ImmutableSetMultimap.Builder<String, String> builder = ImmutableSetMultimap.builder();
      for (Record2<String, String> record : result) {
        builder.put(record.value1(), record.value2());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch preferences by user from %s for %s/%s",
          SET_PREFERENCES.getName(), category, preferenceName), dae);
    }
  }

  @Override
  public ImmutableSet<String> getUsersWhoseSetPreferenceHas(
      String category, String preferenceName, String value) throws IOException {
    try {
      Result<Record1<String>> result = createDslContext().select(SET_PREFERENCES.USERNAME)
          .from(SET_PREFERENCES)
          .where(SET_PREFERENCES.CATEGORY.eq(category))
          .and(SET_PREFERENCES.KEY.eq(preferenceName))
          .and(SET_PREFERENCES.VALUE.eq(value))
          .fetch();

      if (result.isEmpty()) {
        return ImmutableSet.of();
      }

      ImmutableSet.Builder<String> builder = ImmutableSet.builder();
      for (Record1<String> record : result) {
        builder.add(record.value1());
      }

      return builder.build();
    } catch (DataAccessException dae) {
      throw new IOException(String.format(
          "Failed to fetch users from %s for %s/%s which has value %s",
          SET_PREFERENCES.getName(), category, preferenceName, value), dae);
    }
  }

  private DSLContext createDslContext() {
    return DSL.using(dataSource, sqlDialect);
  }
}
