package cloud.toolshed.jsup;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Range;

import java.io.IOException;

/**
 * Provides methods to perform common queries of user preferences.
 *
 * <p>Can be used for analytics, eg. which users have ui.theme set to default,
 * or for interfacing with other parts of the application, eg. which users
 * want to be emailed on failures.
 *
 * <p>For example:
 * <pre>
 * {@code
 * Injector injector = Guice.createInjector(
 *     new AbstractModule() {
 *       @Override
 *       protected void configure() {
 *         HikariConfig config = new HikariConfig();
 *         config.setJdbcUrl("jdbc:h2:mem:testdb");
 *         HikariDataSource dataSource = new HikariDataSource(config);
 *
 *         bind(Key.get(SQLDialect.class, UserPreferencesModule.UserPreferences.class)).toInstance(SQLDialect.H2);
 *         bind(Key.get(DataSource.class, UserPreferencesModule.UserPreferences.class)).toInstance(dataSource);
 *       }
 *     },
 *     new UserPreferencesModule());
 *
 * UserPreferencesDb userPreferencesDb = injector.getInstance(UserPreferencesDb.class);
 * UserPreferencesAnalytics userPreferencesAnalytics = injector.getInstance(UserPreferencesAnalytics.class);
 *
 * userPreferencesDb.setString("some-user", "ui", "theme", "default");
 * userPreferencesDb.setString("some-user", "ui", "display.timestamp.timezone", "UTC");
 * userPreferencesDb.addToSet("some-user", "app", "favorite.projects", ImmutableSet.of("prj1", "prj2"));
 * userPreferencesDb.setBoolean("some-user", "app", "email.on.failure", true);
 *
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "theme"));
 * System.out.println(userPreferencesDb.getString("some-user", "ui", "display.timestamp.timezone"));
 * System.out.println(userPreferencesDb.getSet("some-user", "app", "favorite.projects"));
 * System.out.println(userPreferencesDb.getBoolean("some-user", "app", "email.on.failure"));
 *
 * System.out.println(userPreferencesAnalytics.getStringCategories());
 * System.out.println(userPreferencesAnalytics.getBooleanPreferencesByUser("app", "email.on.failure"));
 * }
 * </pre>
 *
 * @author Akshay Dayal
 */
public interface UserPreferencesAnalytics {

  /**
   * Get all the boolean categories present in the database.
   */
  ImmutableSet<String> getBooleanCategories() throws IOException;

  /**
   * Get all the integer categories present in the database.
   */
  ImmutableSet<String> getIntegerCategories() throws IOException;

  /**
   * Get all the float categories present in the database.
   */
  ImmutableSet<String> getFloatCategories() throws IOException;

  /**
   * Get all the string categories present in the database.
   */
  ImmutableSet<String> getStringCategories() throws IOException;

  /**
   * Get all the set categories present in the database.
   */
  ImmutableSet<String> getSetCategories() throws IOException;

  /**
   * Get all the preference names with values for the given boolean category.
   */
  ImmutableSet<String> getBooleanPreferenceNames(String category) throws IOException;

  /**
   * Get all the preference names with values for the given integer category.
   */
  ImmutableSet<String> getIntegerPreferenceNames(String category) throws IOException;

  /**
   * Get all the preference names with values for the given float category.
   */
  ImmutableSet<String> getFloatPreferenceNames(String category) throws IOException;

  /**
   * Get all the preference names with values for the given string category.
   */
  ImmutableSet<String> getStringPreferenceNames(String category) throws IOException;

  /**
   * Get all the preference names with values for the given set category.
   */
  ImmutableSet<String> getSetPreferenceNames(String category) throws IOException;

  /**
   * Get a map of preference values by username for the given boolean category and preference name.
   */
  ImmutableMap<String, Boolean> getBooleanPreferencesByUser(String category, String preferenceName) throws IOException;

  /**
   * Get the usernames who have a boolean preference with the given value.
   */
  ImmutableSet<String> getUsersWithBooleanPreference(
      String category, String preferenceName, boolean value) throws IOException;

  /**
   * Get a map of preference values by username for the given integer category and preference name.
   */
  ImmutableMap<String, Integer> getIntegerPreferencesByUser(String category, String preferenceName) throws IOException;

  /**
   * Get a map of preference values by username for the given integer category and preference name, where the value
   * must be within the given range.
   */
  ImmutableMap<String, Integer> getIntegerPreferencesByUser(
      String category, String preferenceName, Range<Integer> range) throws IOException;

  /**
   * Get the usernames who have an integer preference with the given value.
   */
  ImmutableSet<String> getUsersWithIntegerPreference(
      String category, String preferenceName, int value) throws IOException;

  /**
   * Get a map of preference values by username for the given float category and preference name.
   */
  ImmutableMap<String, Float> getFloatPreferencesByUser(String category, String preferenceName) throws IOException;

  /**
   * Get a map of preference values by username for the given float category and preference name, where the value
   * must be within the given range.
   */
  ImmutableMap<String, Float> getFloatPreferencesByUser(
      String category, String preferenceName, Range<Float> range) throws IOException;

  /**
   * Get the usernames who have a float preference with the given value.
   */
  ImmutableSet<String> getUsersWithFloatPreference(
      String category, String preferenceName, float value) throws IOException;

  /**
   * Get a map of preference values by username for the given string category and preference name.
   */
  ImmutableMap<String, String> getStringPreferencesByUser(String category, String preferenceName) throws IOException;

  /**
   * Get a map of preference values by username for the given category and preference name, where the value starts with
   * the given string.
   */
  ImmutableMap<String, String> getStringPreferencesByUserStartingWith(
      String category, String preferenceName, String startsWith) throws IOException;

  /**
   * Get a map of preference values by username for the given category and preference name, where the value ends with
   * the given string.
   */
  ImmutableMap<String, String> getStringPreferencesByUserEndingWith(
      String category, String preferenceName, String endsWith) throws IOException;

  /**
   * Get a map of preference values by username for the given category and preference name, where the value contains
   * the given string.
   */
  ImmutableMap<String, String> getStringPreferencesByUserContaining(
      String category, String preferenceName, String containing) throws IOException;

  /**
   * Get the usernames who have a string preference with the given value.
   */
  ImmutableSet<String> getUsersWithStringPreference(
      String category, String preferenceName, String value) throws IOException;

  /**
   * Get a map of preference values by username for the given set category and preference name.
   */
  ImmutableSetMultimap<String, String> getSetPreferencesByUser(
      String category, String preferenceName) throws IOException;

  /**
   * Get the usernames who have a set preference that contains the given value.
   *
   * <p>Example, find all usernames whose "app","favorite.cars" has "BMW".
   */
  ImmutableSet<String> getUsersWhoseSetPreferenceHas(
      String category, String preferenceName, String value) throws IOException;
}
