/*
 Copyright (C) 2017 Electronic Arts Inc.  All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 1.  Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
 2.  Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
 3.  Neither the name of Electronic Arts, Inc. ("EA") nor the names of
     its contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY ELECTRONIC ARTS AND ITS CONTRIBUTORS "AS IS" AND ANY
 EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL ELECTRONIC ARTS OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package cloud.orbit.actors.cluster.impl;

import org.redisson.api.RedissonClient;
import org.redisson.api.listener.MessageListener;

import cloud.orbit.tuples.Pair;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

public class RedisOrbitClient
{
    private Timer timer = new Timer();
    private final RedissonClient redisClient;
    private boolean isConnected = false;
    private final List<Pair<String, MessageListener<Object>>> subscriptions = new ArrayList<>();

    public RedisOrbitClient(final RedissonClient redisClient)
    {
        this.redisClient = redisClient;
        onTimer();

    }

    public void subscribe(final String channelId, final MessageListener<Object> messageListener)
    {
        subscriptions.add(Pair.of(channelId, messageListener));
        if(isConnected)
        {
            redisClient.getTopic(channelId).addListener(messageListener);
        }
    }

    private void onTimer()
    {
        final Boolean nowConnected = redisClient.getNodesGroup().pingAll();

        if(nowConnected && !isConnected)
        {
            subscriptions.forEach((sub) ->
            {
                redisClient.getTopic(sub.getLeft()).removeAllListeners();
                redisClient.getTopic(sub.getLeft()).addListener(sub.getRight());
            });
        }

        isConnected = nowConnected;

        timer = new Timer();
        timer.schedule(new TimerTask()
        {
            @Override
            public void run()
            {
                onTimer();
            }
        }, 10000);
    }

    public boolean isConnectied () {
        return isConnected;
    }

    public RedissonClient getRedissonClient() {
        return this.redisClient;
    }

    public void shutdown() {
        this.redisClient.shutdown();
    }
}
