/*
 Copyright (C) 2015 - 2019 Electronic Arts Inc.  All rights reserved.
 This file is part of the Orbit Project <https://www.orbit.cloud>.
 See license in LICENSE.
 */

package orbit.client

import kotlinx.coroutines.launch
import mu.KotlinLogging
import orbit.client.actor.ActorProxyFactory
import orbit.client.addressable.AddressableDefinitionDirectory
import orbit.client.addressable.AddressableProxyFactory
import orbit.client.addressable.CapabilitiesScanner
import orbit.client.addressable.InvocationSystem
import orbit.client.execution.ExecutionLeases
import orbit.client.execution.ExecutionSystem
import orbit.client.mesh.AddressableLeaser
import orbit.client.mesh.NodeLeaser
import orbit.client.net.ClientAuthInterceptor
import orbit.client.net.ConnectionHandler
import orbit.client.net.GrpcClient
import orbit.client.net.LocalNode
import orbit.client.net.MessageHandler
import orbit.client.serializer.Serializer
import orbit.util.concurrent.SupervisorScope
import orbit.util.di.ComponentContainer
import orbit.util.time.Clock
import orbit.util.time.ConstantTicker
import orbit.util.time.stopwatch
import kotlin.coroutines.CoroutineContext

class OrbitClient(val config: OrbitClientConfig = OrbitClientConfig()) {
    private val logger = KotlinLogging.logger { }

    private val container = ComponentContainer()
    val clock = Clock()

    private val scope = SupervisorScope(
        pool = config.pool,
        exceptionHandler = this::onUnhandledException
    )

    private val ticker = ConstantTicker(
        scope = scope,
        targetTickRate = config.tickRate.toMillis(),
        clock = clock,
        logger = logger,
        exceptionHandler = this::onUnhandledException,
        autoStart = false,
        onTick = this::tick
    )

    init {
        container.configure {
            instance(this@OrbitClient)
            instance(config)
            instance(scope)
            instance(clock)
            instance(LocalNode(config))

            definition<GrpcClient>()
            definition<ClientAuthInterceptor>()
            definition<ConnectionHandler>()
            definition<MessageHandler>()

            definition<NodeLeaser>()
            definition<AddressableLeaser>()

            definition<Serializer>()

            definition<CapabilitiesScanner>()
            definition<AddressableProxyFactory>()
            definition<InvocationSystem>()
            definition<AddressableDefinitionDirectory>()
            externallyConfigured(config.addressableConstructor)


            definition<ExecutionSystem>()
            definition<ExecutionLeases>()

            definition<ActorProxyFactory>()
        }
    }

    private val nodeLeaser by container.inject<NodeLeaser>()
    private val messageHandler by container.inject<MessageHandler>()

    private val connectionHandler by container.inject<ConnectionHandler>()
    private val capabilitiesScanner by container.inject<CapabilitiesScanner>()
    private val localNode by container.inject<LocalNode>()
    private val definitionDirectory by container.inject<AddressableDefinitionDirectory>()
    private val executionSystem by container.inject<ExecutionSystem>()

    val actorFactory by container.inject<ActorProxyFactory>()

    fun start() = scope.launch {
        logger.info("Starting Orbit client...")
        val (elapsed, _) = stopwatch(clock) {
            // Scan for capabilities
            capabilitiesScanner.scan()
            definitionDirectory.setupDefinition(
                interfaceClasses = capabilitiesScanner.addressableInterfaces,
                impls = capabilitiesScanner.interfaceLookup
            )
            localNode.manipulate {
                it.copy(capabilities = definitionDirectory.generateCapabilities())
            }

            // Get first lease
            nodeLeaser.joinCluster()

            // Open message channel
            connectionHandler.connect()

            // Start tick
            ticker.start()
        }

        logger.info("Orbit client started successfully in {}ms.", elapsed)
    }

    private suspend fun tick() {
        // See if lease needs renewing
        nodeLeaser.tick()

        // Timeout messages etc
        messageHandler.tick()

        // Handle actor deactivations and leases
        executionSystem.tick()
    }

    fun stop() = scope.launch {
        logger.info("Stopping Orbit...")
        val (elapsed, _) = stopwatch(clock) {
            // Stop messaging
            connectionHandler.disconnect()

            // Stop the tick
            ticker.stop()
        }

        logger.info("Orbit stopped successfully in {}ms.", elapsed)
    }

    @Suppress("UNUSED_PARAMETER")
    private fun onUnhandledException(coroutineContext: CoroutineContext, throwable: Throwable) =
        onUnhandledException(throwable)

    private fun onUnhandledException(throwable: Throwable) {
        logger.error(throwable) { "Unhandled exception in Orbit Client." }
    }

}