/*
 * Copyright 2018 Emory University
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cloud.elit.nlp.component.token;

import cloud.elit.sdk.DecodeComponent;
import cloud.elit.sdk.nlp.component.Tokenizer;
import cloud.elit.sdk.nlp.structure.Document;
import cloud.elit.sdk.nlp.structure.Sentence;

import java.io.BufferedReader;
import java.io.FileReader;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class SimpleTokenizer extends Tokenizer<SimpleTokenizerParameters> {
    private Pattern DELIM;

    public SimpleTokenizer() {
        DELIM = Pattern.compile("\\s+");
    }

    @Override
    public void load(String model_path, SimpleTokenizerParameters params) {
        try {
            BufferedReader reader = new BufferedReader(new FileReader(model_path+"/"+params.getModelName()));
            String line;

            for (int i=0; (line = reader.readLine()) != null; i++) {
                if (i == params.getChoice()) {
                    DELIM = Pattern.compile(line.trim());
                    break;
                }
            }
        } catch (Exception e) { e.printStackTrace(); }
    }

    @Override
    public Document decode(String input, SimpleTokenizerParameters params) {
        String[] tokens = DELIM.split(input);
        Document document = new Document();
        int begin = 0;

        for (int end=0; end<tokens.length; end++) {
            String token = tokens[end];
            if (token.endsWith(".") || token.endsWith("?") || token.endsWith("!")) {
                List<String> t = IntStream.range(begin, end+1).mapToObj(j -> tokens[j]).collect(Collectors.toList());
                document.addSentence(new Sentence(getSubList(tokens, begin, end+1)));
                begin = end + 1;
            }
        }

        if (begin < tokens.length)
            document.addSentence(new Sentence(getSubList(tokens, begin, tokens.length)));

        return document;
    }

    @Override
    public void save(String model_path, SimpleTokenizerParameters params) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void train(List<String> trn_data, List<String> dev_data, SimpleTokenizerParameters params) {
        throw new UnsupportedOperationException();
    }

    /**
     * @param array
     * @param begin inclusive.
     * @param end exclusive.
     * @return
     */
    private List<String> getSubList(String[] array, int begin, int end) {
        return IntStream.range(begin, end).mapToObj(i -> array[i]).collect(Collectors.toList());
    }
}
