/*
 * Decompiled with CFR 0.152.
 */
package com.cirrustech.publisher.callable;

import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.cloudwatch.model.MetricDatum;
import com.amazonaws.services.cloudwatch.model.PutMetricDataRequest;
import com.cirrustech.publisher.callable.PropertiesFileHandler;
import com.cirrustech.utils.FileUtils;
import com.cirrustech.utils.JSONUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedCallable
implements Callable {
    private static final Logger LOG = LoggerFactory.getLogger(FileBasedCallable.class);
    private static final int MAX_ITEMS_PER_BATCH = 18;
    private static final int DEFAULT_TIME_TO_KEEP_FILES = 10800000;
    private final String filePath;
    private final AmazonCloudWatch client;
    private final int millisBetweenRun;
    private final String namespace;
    private long maxTimeToKeepFilesInMillis = 10800000L;

    public FileBasedCallable(AmazonCloudWatch client, String filePath, String namespace, int millisBetweenRun) {
        Preconditions.checkNotNull((Object)client, (Object)"CW client cannot be null or empty.");
        Preconditions.checkNotNull((Object)StringUtils.trimToNull((String)filePath), (Object)"File path cannot be null or empty.");
        Preconditions.checkNotNull((Object)StringUtils.trimToNull((String)namespace), (Object)"Namespace cannot be null or empty.");
        Preconditions.checkArgument((millisBetweenRun > 0 ? 1 : 0) != 0, (Object)"Wait time cannot be negative.");
        this.filePath = filePath;
        this.client = client;
        this.millisBetweenRun = millisBetweenRun;
        this.namespace = namespace;
    }

    public Object call() throws Exception {
        while (true) {
            this.run();
        }
    }

    @VisibleForTesting
    void run() throws IOException {
        for (File file : this.getAllFiles()) {
            if (!file.isDirectory() && FileUtils.isLogFile(file)) {
                try {
                    String propertiesFileName = FileUtils.generatePropertiesFileNameForLogFile(file.getName());
                    PropertiesFileHandler handler = new PropertiesFileHandler(this.filePath + propertiesFileName);
                    if (this.fileIsProcessed(file, handler)) {
                        LOG.info("File {} is completely processed.", (Object)file);
                        if (!this.checkNoRecentUpdate(handler.getLastUpdateTimestamp())) continue;
                        LOG.info("Delete old file {}", (Object)file);
                        this.deleteFile(file, handler);
                        continue;
                    }
                    LOG.info("Processing file {}", (Object)file);
                    this.processFile(file, handler);
                }
                catch (IOException e) {
                    LOG.warn("Exception on processing file {} {}", (Object)file, (Object)e);
                }
                continue;
            }
            LOG.info("Not a log file {}. Skipping.", (Object)file);
        }
        this.sleep(this.millisBetweenRun);
    }

    public void setMaxTimeToKeepFilesInMillis(long maxTimeToKeepFilesInMillis) {
        Preconditions.checkArgument((maxTimeToKeepFilesInMillis > 0L ? 1 : 0) != 0, (Object)"Retention period cannot be negative");
        this.maxTimeToKeepFilesInMillis = maxTimeToKeepFilesInMillis;
    }

    private void sleep(int maxMillisToWait) throws IOException {
        try {
            Thread.sleep(maxMillisToWait);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
    }

    private void processFile(File file, PropertiesFileHandler handler) throws IOException {
        try (BufferedReader reader = new BufferedReader(new FileReader(file));){
            String line;
            reader.skip(handler.getBytesRead());
            int bytesRead = 0;
            ArrayList metricDatums = Lists.newArrayList();
            while ((line = reader.readLine()) != null) {
                boolean result;
                bytesRead += line.getBytes().length + 1;
                MetricDatum datum = JSONUtils.decodeJSON(line);
                if (datum != null) {
                    metricDatums.add(datum);
                }
                if (metricDatums.size() < 18 || !(result = this.tryPushMetricToCW(metricDatums))) continue;
                handler.updateRecords(bytesRead);
                bytesRead = 0;
                metricDatums.clear();
            }
            boolean result = this.tryPushMetricToCW(metricDatums);
            if (result) {
                handler.updateRecords(bytesRead);
            }
        }
    }

    private boolean tryPushMetricToCW(List<MetricDatum> metricDatums) {
        PutMetricDataRequest request = new PutMetricDataRequest().withNamespace(this.namespace).withMetricData(metricDatums);
        try {
            if (!metricDatums.isEmpty()) {
                this.client.putMetricData(request);
                LOG.info("Published {} metrics in a batch into CW.", (Object)metricDatums.size());
            }
            return true;
        }
        catch (Exception e) {
            LOG.warn("Exception on publishing metrics into CW {}", (Throwable)e);
            return false;
        }
    }

    private void deleteFile(File file, PropertiesFileHandler handler) {
        file.delete();
        handler.delete();
        LOG.info("Deleted file: {}", (Object)file.getName());
    }

    private boolean fileIsProcessed(File file, PropertiesFileHandler handler) {
        return handler.propertiesFileExists() && file.length() == (long)handler.getBytesRead();
    }

    private boolean checkNoRecentUpdate(long lastUpdateTimestamp) {
        return System.currentTimeMillis() - lastUpdateTimestamp > this.maxTimeToKeepFilesInMillis;
    }

    private List<File> getAllFiles() {
        File[] files = new File(this.filePath).listFiles();
        if (files == null) {
            return Lists.newArrayList();
        }
        List<File> theFiles = Arrays.asList(files);
        Collections.sort(theFiles);
        return ImmutableList.copyOf(theFiles);
    }
}

