(ns clojurewerkz.elastisch.native.document
  (:refer-clojure :exclude [get replace count sort])
  (:require [clojurewerkz.elastisch.native :as es]
            [clojurewerkz.elastisch.native.conversion :as cnv]
            [clojurewerkz.elastisch.query  :as q]
            [clojurewerkz.elastisch.native.response :as r])
  (:import clojure.lang.IPersistentMap
           [org.elasticsearch.action.get GetResponse MultiGetResponse]
           org.elasticsearch.action.count.CountResponse
           org.elasticsearch.action.delete.DeleteResponse
           org.elasticsearch.action.deletebyquery.DeleteByQueryResponse
           org.elasticsearch.action.search.SearchResponse
           java.util.Map))

(defn ^IPersistentMap create
  "Adds document to the search index and waits for the response.
   If not given as an option, document id will be generated automatically.

   Options:

     * :id (string): unique document id. If not provided, it will be generated by ElasticSearch
     * :timestamp (string): document timestamp either as millis since the epoch,
                                          or, in the configured date format
     * :ttl (long): document TTL in milliseconds. Must be > 0
     * :refresh (boolean, default: false): should a refresh be executed post this index operation?
     * :version (long): document version
     * :version-type (string, default: \"internal\"): \"internal\" or \"external\"
     * :content-type (string): document content type
     * :routing (string): controls the shard routing of the request. Using this value to hash the shard
                          and not the id
     * :percolate (string): the percolate query to use to reduce the percolated queries that are going to run against this doc.
                           Can be set to \"*\" which means \"all queries\"
     * :parent (string): parent document id

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])

   (doc/create \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28})

   (doc/create \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28} :id \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  ([index mapping-type document]
     (let [res (es/index (cnv/->index-request index
                                              mapping-type
                                              document
                                              {:op-type "create"}))]
       (cnv/index-response->map (.actionGet res))))
  ([index mapping-type document & {:as params}]
     (let [res (es/index (cnv/->index-request index
                                              mapping-type
                                              document
                                              (merge params {:op-type "create"})))]
       (cnv/index-response->map (.actionGet res)))))

(defn async-create
  "Adds document to the search index and returns a future without waiting
    for the response. Takes exactly the same arguments as create."
  ([index mapping-type document]
     (future (create index mapping-type document)))
  ([index mapping-type document & {:as params}]
     (future (apply create (concat [index mapping-type document] params)))))

(defn put
  "Creates or updates a document in the search index using the provided document id
   and waits for the response."
  ([index mapping-type id document]
     (let [res (es/index (cnv/->index-request index
                                              mapping-type
                                              document
                                              {:id id :op-type "index"}))]
       (cnv/index-response->map (.actionGet res))))
  ([index mapping-type id document & {:as params}]
     (let [res (es/index (cnv/->index-request index
                                              mapping-type
                                              document
                                              (merge params {:id id :op-type "index"})))]
       (cnv/index-response->map (.actionGet res)))))

(defn async-put
  "Creates or updates a document in the search index using the provided document id
   and returns a future without waiting for the response. Takes exactly the same arguments as put."
  ([index mapping-type id document]
     (future (put index mapping-type id document)))
  ([index mapping-type id document & {:as params}]
     (future (apply put (concat [index mapping-type id document] params)))))

(defn upsert
  "Updates or creates a document using provided data"
  ([index mapping-type ^String id ^Map doc]
    (let [res (es/update (cnv/->upsert-request index
                                               mapping-type
                                               id
                                               doc))]
      (cnv/update-response->map (.actionGet res)))))

(defn update-with-script
  "Updates a document using a script"
  ([index mapping-type ^String id ^String script]
     (let [res (es/update (cnv/->update-request index
                                                mapping-type
                                                id
                                                script))]
       (cnv/update-response->map (.actionGet res))))
  ([index mapping-type ^String id ^String script ^Map params]
     (let [res (es/update (cnv/->update-request index
                                                mapping-type
                                                id
                                                script
                                                params))]
       (cnv/update-response->map (.actionGet res)))))


(defn get
  "Fetches and returns a document by id or nil if it does not exist.
   Waits for response.

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])

   (doc/get \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  ([index mapping-type id]
     (let [ft               (es/get (cnv/->get-request index
                                                       mapping-type
                                                       id))
           ^GetResponse res (.actionGet ft)]
       (when (.isExists res)
         (cnv/get-response->map res))))
  ([index mapping-type id & {:as params}]
     (let [ft               (es/get (cnv/->get-request index
                                                       mapping-type
                                                       id
                                                       params))
           ^GetResponse res (.actionGet ft)]
       (when (.isExists res)
         (cnv/get-response->map (.actionGet ft))))))

(defn async-get
  "Fetches and returns a document by id or nil if it does not exist.
   Returns a future without waiting."
  ([index mapping-type id]
     (future (get index mapping-type id)))
  ([index mapping-type id & {:as params}]
     (future (apply get (concat [index mapping-type id] params)))))

(defn present?
  "Returns true if a document with the given id is present in the provided index
   with the given mapping type."
  [index mapping-type id]
  (not (nil? (get index mapping-type id))))

(defn multi-get
  "Multi get returns only documents that are found (exist).

   Queries can passed as a collection of maps with three keys: :_index,
   :_type and :_id:

   (doc/multi-get [{:_index index-name :_type mapping-type :_id \"1\"}
                   {:_index index-name :_type mapping-type :_id \"2\"}])


   2-argument version accepts an index name that eliminates the need to include
   :_index in every query map:

   (doc/multi-get index-name [{:_type mapping-type :_id \"1\"}
                              {:_type mapping-type :_id \"2\"}])

   3-argument version also accepts a mapping type that eliminates the need to include
   :_type in every query map:

   (doc/multi-get index-name mapping-type [{:_id \"1\"}
                                           {:_id \"2\"}])"
  ([queries]
     ;; example response from the REST API:
     ;; ({:_index people, :_type person, :_id 1, :_version 1, :exists true, :_source {...}})
     (let [ft                    (es/multi-get (cnv/->multi-get-request queries))
           ^MultiGetResponse res (.actionGet ft)
           results               (cnv/multi-get-response->seq res)]
       (filter :exists results)))
  ([index queries]
     (let [qs                    (map #(assoc % :_index index) queries)
           ft                    (es/multi-get (cnv/->multi-get-request qs))
           ^MultiGetResponse res (.actionGet ft)
           results               (cnv/multi-get-response->seq res)]
       (filter :exists results)))
  ([index mapping-type queries]
     (let [qs                    (map #(assoc % :_index index :_type mapping-type) queries)
           ft                    (es/multi-get (cnv/->multi-get-request qs))
           ^MultiGetResponse res (.actionGet ft)
           results               (cnv/multi-get-response->seq res)]
       (filter :exists results))))

(defn delete
  "Deletes document from the index.

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])

   (doc/delete \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  ([index mapping-type id]
     (let [ft                  (es/delete (cnv/->delete-request index mapping-type id))
           ^DeleteResponse res (.actionGet ft)]
       (cnv/delete-response->map res)))
  ([index mapping-type id & {:as options}]
     (let [ft                  (es/delete (cnv/->delete-request index mapping-type id options))
           ^DeleteResponse res (.actionGet ft)]
       (cnv/delete-response->map res))))

(defn delete-by-query
  "Performs a delete-by-query operation.

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/delete-by-query \"people\" \"person\" (q/term :username \"appleseed\"))"
  ([index mapping-type query]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request index mapping-type query))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res)))
  ([index mapping-type query & {:as options}]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request index mapping-type query options))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res))))

(defn delete-by-query-across-all-types
  "Performs a delete-by-query operation across all mapping types."
  ([index query]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request-across-all-types index query))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res)))
  ([index query & {:as options}]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request-across-all-types index query options))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res))))

(defn delete-by-query-across-all-indexes-and-types
  "Performs a delete-by-query operation across all indexes and mapping types.
   This may put very high load on your ElasticSearch cluster so use this function with care."
  ([query]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request-across-all-indices-and-types query))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res)))
  ([query & {:as options}]
     (let [ft                         (es/delete-by-query (cnv/->delete-by-query-request-across-all-indices-and-types query options))
           ^DeleteByQueryResponse res (.actionGet ft)]
       (cnv/delete-by-query-response->map res))))

(defn count
  "Performs a count query.

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/count \"people\" \"person\")
   (doc/count \"people\" \"person\" (q/prefix :username \"appl\"))"
  ([index mapping-type]
     (count index mapping-type (q/match-all)))
  ([index mapping-type query]
     (let [ft (es/count (cnv/->count-request index mapping-type {:query query}))
           ^CountResponse res (.actionGet ft)]
       (merge {:count (.getCount res)}
              (cnv/broadcast-operation-response->map res))))
  ([index mapping-type query & {:as options}]
     (let [ft (es/count (cnv/->count-request index mapping-type (merge options
                                                                       {:query query})))
           ^CountResponse res (.actionGet ft)]
       (merge {:count (.getCount res)}
              (cnv/broadcast-operation-response->map res)))))

(defn search
  "Performs a search query across one or more indexes and one or more mapping types.

   Examples:

   (require '[clojurewerkz.elastisch.native.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/search \"people\" \"person\" :query (q/prefix :username \"appl\"))"
  [index mapping-type & {:as options}]
  (let [ft                  (es/search (cnv/->search-request index mapping-type options))
        ^SearchResponse res (.actionGet ft)]
    (cnv/search-response->seq res)))

(defn search-all-types
  "Performs a search query across one or more indexes and all mapping types."
  [index & {:as options}]
  (let [ft                  (es/search (cnv/->search-request index nil options))
        ^SearchResponse res (.actionGet ft)]
    (cnv/search-response->seq res)))

(defn search-all-indexes-and-types
  "Performs a search query across all indexes and all mapping types. This may put very high load on your
   ElasticSearch cluster so use this function with care."
  [& {:as options}]
  (let [ft                  (es/search (cnv/->search-request [] nil options))
        ^SearchResponse res (.actionGet ft)]
    (cnv/search-response->seq res)))

(defn scroll
  "Performs a scroll query, fetching the next page of results from a
   query given a scroll id"
  [scroll-id & {:as options}]
  (let [ft                  (es/search-scroll (cnv/->search-scroll-request scroll-id options))
        ^SearchResponse res (.actionGet ft)]
    (cnv/search-response->seq res)))

(defn scroll-seq
  "Returns a lazy sequence of all documents for a given scroll query"
  [prev-resp]
  (let [hits      (r/hits-from prev-resp)
        scroll-id (:_scroll_id prev-resp)]
    (if (seq hits)
      (concat hits (lazy-seq (scroll-seq (scroll scroll-id :scroll "1m"))))
      hits)))

(defn replace
  "Replaces document with given id with a new one"
  [index mapping-type id document]
  (delete index mapping-type id)
  (put index mapping-type id document))

(defn more-like-this
  "Performs a More Like This (MLT) query."
  [index mapping-type id &{:as options}]
  (let [ft                  (es/more-like-this (cnv/->more-like-this-request index mapping-type id options))
        ^SearchResponse res (.actionGet ft)]
    (cnv/search-response->seq res)))
