package ch.inftec.ju.util;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Builder to create PropertyChain instances.
 * <p>
 * A PropertyChain can evaluate properties using multiple PropertyEvaluators that can
 * be arranged in a chain, priorizing the property evaluation by the order they are added to
 * the chain.
 * 
 * @author Martin
 *
 */
public class PropertyChainBuilder {
	private Logger logger = LoggerFactory.getLogger(PropertyChainBuilder.class);
	
	private final List<PropertyEvaluator> evaluators = new ArrayList<>();
	
	// Attributes of the PropertyChain
	private boolean defaultThrowExceptionIfUndefined = false;
	
	/**
	 * Adds an evaluator that evaluates system properties.
	 * @return This builder to allow for chaining
	 */
	public PropertyChainBuilder addSystemPropertyEvaluator() {
		return this.addPropertyEvaluator(new SystemPropertyEvaluator());
	}
	
	/**
	 * Adds an evaluator that reads properties from property files.
	 * @param resourceUrl URL to property file resource
	 * @return This builder to allow for chaining
	 */
	public PropertyChainBuilder addResourcePropertyEvaluator(URL resourceUrl) {
		try {
			return this.addPropertyEvaluator(new PropertiesPropertyEvaluator(resourceUrl));
		} catch (JuException ex) {
			throw new JuRuntimeException("Couldn't load properties from url " + resourceUrl, ex);
		}
	}
	
	/**
	 * Adds an evaluator that reads properties from a property file.
	 * @param resourceName Name of the resource
	 * @param ignoreMissingResource If true, no evalautor will be added and no exception will
	 * be thrown if the resource doesn't exist
	 * @return This builder to allow for chaining
	 */
	public PropertyChainBuilder addResourcePropertyEvaluator(String resourceName, boolean ignoreMissingResource) {
		try {
			URL resourceUrl = JuUrl.resource(resourceName);
			return this.addPropertyEvaluator(new PropertiesPropertyEvaluator(resourceUrl));
		} catch (JuException ex) {
			if (ignoreMissingResource) {
				logger.debug(String.format("Ignoring missing resource %s (Exception: %s)", resourceName, ex.getMessage()));
				return this;
			} else {
				throw new JuRuntimeException("Couldn't load properties from resource " + resourceName, ex);
			}
		}
	}
	
	/**
	 * Adds an evaluator that reads properties from a CSV (comma separated value) file.
	 * <p>
	 * The profile name will map to the column we want to use.
	 * @param resourceUrl Resource URL to the csv resource
	 * @param profileName Name of the profile (i.e. column identified by it's header / first row) to be used
	 * @param defaultColumnName Name of the column that contains default values if a value is not defined in the profile column
	 * @return
	 */
	public PropertyChainBuilder addCsvPropertyEvaluator(URL resourceUrl, String profileName, String defaultColumnName) {
		CsvPropertyEvaluator csvEvaluator = new CsvPropertyEvaluator(resourceUrl, profileName, defaultColumnName);
		return this.addPropertyEvaluator(csvEvaluator);
	}
	
	/**
	 * Adds a custom implementation of a PropertyEvaluator.
	 * @param evaluator PropertyEvaluator implementation
	 * @return This builder to allow for chaining
	 */
	public PropertyChainBuilder addPropertyEvaluator(PropertyEvaluator evaluator) {
		this.evaluators.add(evaluator);
		return this;
	}

	/**
	 * Sets the default exception throwing behaviour if a property is undefined.
	 * <p>
	 * Initial value is false, i.e. no exceptions are thrown if a property is undefined and null
	 * is returned.
	 * @param defaultThrowExceptionIfUndefined True if by default, an exception should be thrown if a property is undefined
	 * @return This builder to allow for chaining
	 */
	public PropertyChainBuilder setDefaultThrowExceptionIfUndefined(boolean defaultThrowExceptionIfUndefined) {
		this.defaultThrowExceptionIfUndefined = defaultThrowExceptionIfUndefined;
		return this;
	}
	
	/**
	 * Gets the PropertyChain that was built using this builder.
	 * @return PropertyChain instance
	 */
	public PropertyChain getPropertyChain() {
		return new PropertyChainImpl();
	}
	
	private class PropertyChainImpl implements PropertyChain {

		@Override
		public String get(String key) {
			return this.get(key, defaultThrowExceptionIfUndefined);
		}

		@Override
		public String get(String key, boolean throwExceptionIfNotDefined) {
			Object obj = this.getObject(key, null, throwExceptionIfNotDefined);
			return obj == null ? null : obj.toString();
		}

		@Override
		public String get(String key, String defaultValue) {
			String val = this.get(key, false);
			return val != null ? val : defaultValue;
		}
		
		@Override
		public <T> T get(String key, Class<T> clazz) {
			String val = this.get(key);
			return this.convert(val, clazz);
		}
		
		@Override
		public <T> T get(String key, Class<T> clazz, boolean throwExceptionIfNotDefined) {
			String val = this.get(key, throwExceptionIfNotDefined);
			return this.convert(val, clazz);
		}
		
		@Override
		public <T> T get(String key, Class<T> clazz, String defaultValue) {
			String val = this.get(key, defaultValue);
			return this.convert(val, clazz);
		}
		
		@SuppressWarnings("unchecked")
		private <T> T convert(String val, Class<T> clazz) {
			if (StringUtils.isEmpty(val)) return null;
			
			if (clazz == Integer.class) {
				return (T) new Integer(val);
			} else if (clazz == Boolean.class) {
				return (T) new Boolean(val);
			} else {
				throw new JuRuntimeException("Conversion not supported: " + clazz);
			}
		}
		
		private Object getObject(String key, Object defaultValue, boolean throwExceptionIfNotDefined) {
			Object obj = this.evaluate(key);
			if (obj == null) {
				if (throwExceptionIfNotDefined) {
					throw new JuRuntimeException("Property undefined: " + key);
				} else {
					return defaultValue;
				}
			}
			return obj;
		}
		
		private Object evaluate(String key) {
			for (PropertyEvaluator evaluator : evaluators) {
				Object val = evaluator.get(key);
				if (val != null) {
					logger.debug("Evaluated property {}={} [using {}]"
							, new Object[] { key, val, evaluator.toString() });
					return val;
				}
			}
			return null;
		}
	}
	
	private static class SystemPropertyEvaluator implements PropertyEvaluator {
		@Override
		public Object get(String key) {
			return System.getProperty(key);
		};
		
		@Override
		public String toString() {
			return JuStringUtils.toString(this);
		}
	}
	
	private static class PropertiesPropertyEvaluator implements PropertyEvaluator {
		private final URL propertiesUrl;
		private final Properties props;
		
		public PropertiesPropertyEvaluator(URL propertiesUrl) throws JuException {
			this.propertiesUrl = propertiesUrl;
			this.props = new IOUtil().loadPropertiesFromUrl(propertiesUrl);
		}
		
		@Override
		public Object get(String key) {
			return this.props.get(key);
		};
		
		@Override
		public String toString() {
			return JuStringUtils.toString(this, "url", this.propertiesUrl);
		}
	}
	
	private static class CsvPropertyEvaluator implements PropertyEvaluator {
		private final URL resourceUrl;
		private final String profile;
		private final CsvTableLookup csvTable;
		
		public CsvPropertyEvaluator(URL resourceUrl, String profile, String defaultColumn) {
			this.resourceUrl = resourceUrl;
			this.profile = profile;
			this.csvTable = CsvTableLookup.build()
					.from(resourceUrl)
					.defaultColumn(defaultColumn)
					.create();
		}
		
		@Override
		public Object get(String key) {
			return this.csvTable.get(key, this.profile);
		}
		
		@Override
		public String toString() {
			return JuStringUtils.toString(this
					, "url", this.resourceUrl
					, "profile", this.profile);
		}
	}
}
