package ch.inftec.ju.util;

import java.io.BufferedReader;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class containing functions related to the JU library itself.
 * @author Martin
 *
 */
public class JuUtils {
	private static Logger logger = LoggerFactory.getLogger(JuUtils.class);
	
	private static PropertyChain juPropertyChain;
	
	/**
	 * Name of the files that contains the property files references.
	 */
	private static String PROPERTIES_FILES_NAME = "ju.properties.files";
	
	/**
	 * Gets a PropertyChain to evaluate ju properties.
	 * <p>
	 * Evaluation of the properties is as follows:
	 * <ol>
	 *   <li><b>ju_module.properties</b>: Optional file to override properties in a module that we don't wan't
	 *       to override even with system properties. Is intended to be checked into source control.</li>
	 *   <li><b>System Property</b>: Overrides all other properties</li>
	 *   <li><b>ju_module_default.properties</b>: Optional file to override default properties in a module. Is intended
	 *       to be checked into source control. In this file, we should only add properties for
	 *       which we want to change the default value for a module (or whatever scope the
	 *       classpath has).</li>
	 *   <li><b>ju_user.properties</b>: Optional user properties file overriding default properties. 
	 *       Must be on the classpath and is not intended to be checked into source control</li>
	 *   <li><b>ju_default.properties</b>: Global default properties that are used when no 
	 *       overriding properties are specified or found. This file also contains the description
	 *       for all possible properties.</li>
	 * </ol>
	 * The PropertyChain of this method is cached, so it is loaded only the first time
	 * it is accessed.
	 * <p>
	 * The PropertyChain is configured <i>not</i> to throw exceptions by default if a property
	 * is undefined.
	 * 
	 * @return PropertyChain implemenation to evaluate JU properties
	 */
	public static PropertyChain getJuPropertyChain() {
		if (juPropertyChain == null) {
			logger.debug("Initializing JU PropertyChain");
			
			List<URL> propFiles = JuUrl.resource().getAll(JuUtils.PROPERTIES_FILES_NAME);
			
			// Process contents of prop files
			
			XString duplicatePrios = new XString();
			Map<Integer, String[]> props = new TreeMap<>();
			for (URL propFile : propFiles) {
				logger.debug("Processing property file: " + propFile);
				
				XString filteredContents = new XString("Filtered contents: " );
				filteredContents.increaseIndent();
				
				try (BufferedReader r = new IOUtil().createReader(propFile)) {
					String line = r.readLine();
					while (line != null) {
						String lineParts[] = JuStringUtils.split(line, ",", true);
						if (lineParts.length > 0 && !lineParts[0].startsWith("#")) {
							AssertUtil.assertTrue("Invalid line: " + line, lineParts.length > 1);
							// Process line
							filteredContents.addLine(line);
							
							int priorization = Integer.parseInt(lineParts[0]);
							if (props.containsKey(priorization)) {
								duplicatePrios.addLineFormatted("Duplicate priorization in %s: %d", JuUtils.PROPERTIES_FILES_NAME, priorization);
							}
							
							props.put(priorization, Arrays.copyOfRange(lineParts, 1, lineParts.length));
						} else {
							// Ignore line
						}
						
						line = r.readLine();
					}
				} catch (Exception ex) {
					throw new JuRuntimeException("Couldn't process property file %s", ex, propFile);
				}
				
				logger.debug(filteredContents.toString());
				
				if (!duplicatePrios.isEmpty()) {
					throw new JuRuntimeException(duplicatePrios.toString());
				}
			}
			
			// Build property chain from read info
			
			PropertyChainBuilder chainBuilder = new PropertyChainBuilder();
			
			XString chainInfo = new XString("Evaluated property chain:");
			chainInfo.increaseIndent();
			for (int prio : props.keySet()) {
				chainInfo.addLine(prio + ": ");
				
				// Get the remaining line parts (without priorization)
				String[] lineParts = props.get(prio);
				
				String propType = lineParts[0];
				// Perform placeholder substitution
				for (int i = 1; i < lineParts.length; i++) {				
					XString part = new XString(lineParts[i]);
					// Substitute %propertyKey% with the appropriate values...
					for (String propertyKey : part.getPlaceHolders()) {
						String val = chainBuilder.getPropertyChain().get(propertyKey);
						if (val != null) {
							part.setPlaceholder(propertyKey, val);
						} else {
							logger.debug("Couldn't replace placeholder: " + propertyKey);
						}
					}
					lineParts[i] = part.toString();
				}
				
				if ("sys".equals(propType)) {
					chainBuilder.addSystemPropertyEvaluator();
					chainInfo.addText("System Properties");
				} else if ("prop".equals(propType)) {
					AssertUtil.assertTrue("prop property type must be followed by a resource path", lineParts.length > 1);
					String resourcePath = lineParts[1];
					boolean optional = lineParts.length > 2 && "optional".equals(lineParts[2]);
					
					URL resourceUrl = JuUrl.resource().single().exceptionIfNone(!optional).get(resourcePath);
					if (resourceUrl != null) {
						chainBuilder.addResourcePropertyEvaluator(resourceUrl);
						chainInfo.addText("Properties file: " + resourceUrl);
					} else {
						AssertUtil.assertTrue("Mandatory resource not found: " + resourcePath, optional);
						chainInfo.addText("Properties file:   >>> optional resource not found: " + resourcePath);
					}
				} else if ("csv".equals(propType)) {
					AssertUtil.assertTrue(
							"prop property type must be followed by a resource path and a profile property name", lineParts.length > 2);
					String resourcePath = lineParts[1];
					String profilePropertyName = lineParts[2];
					
					String profileName = chainBuilder.getPropertyChain().get(profilePropertyName);
					String defaultColumn = lineParts.length > 3
							? lineParts[2]
							: "default";
					
					URL resourceUrl = JuUrl.singleResource(resourcePath);
					chainBuilder.addCsvPropertyEvaluator(resourceUrl, profileName, defaultColumn);
					chainInfo.addFormatted("CSV Properties: %s [profileName=%s, defaultColumn=%s]"
							, resourceUrl
							, profileName
							, defaultColumn);
				} else {
					throw new JuRuntimeException("Unsupported property type: " + propType);
				}
			}
			
			juPropertyChain = chainBuilder.getPropertyChain();
			
			logger.info(chainInfo.toString());
		}
		
		return juPropertyChain;
	}
	
	/**
	 * Clears the cached property chain, forcing a reload.
	 */
	public static void clearPropertyChain() {
		juPropertyChain = null;
	}
}
