package ch.inftec.ju.util.xml;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import ch.inftec.ju.util.JuRuntimeException;

/**
 * Helper class to build XMLs based on DOM (Document Object Model).
 * <p>
 * By default, the generated Document will be set as standalone=yes.
 * @author tgdmemae
 *
 */
public class XmlBuilder {
	private final XmlBuilder parentBuilder;
	private final Document document;
	private final Element element;
	
	static XmlBuilder createRootBuilder(String rootElementName) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder docBuilder = factory.newDocumentBuilder();
			Document document = docBuilder.newDocument();
			document.setXmlStandalone(true);
			Element rootElement = document.createElement(rootElementName);
			document.appendChild(rootElement);
			
			return new XmlBuilder(null, document, rootElement);
		} catch (Exception ex) {
			throw new JuRuntimeException("Couldn't initialize XML Document");
		}
	}
	
	private XmlBuilder(XmlBuilder parentBuilder, Document document, Element element) throws JuRuntimeException {
		this.parentBuilder = parentBuilder;
		this.document = document;
		this.element = element;
	}
	
	/**
	 * Sets an attribute for the current element.
	 * @param name Attribute name
	 * @param value Attribute value
	 * @return This builder to allow for chaining
	 */
	public XmlBuilder setAttribute(String name, String value) {
		this.element.setAttribute(name,  value);
		return this;
	}
	
	/**
	 * Adds text to the current element.
	 * @param text Text to be added
	 * @return This builder to allow for chaining
	 */
	public XmlBuilder addText(String text) {
		Text textNode = this.document.createTextNode(text);
		this.element.appendChild(textNode);
		return this;
	}
	
	/**
	 * Adds a child element to the current element.
	 * <p>
	 * Call endChild as soon as the child node has been configured
	 * @param elementName Name of the child element
	 * @return XmlBuilder to modify the child element
	 */
	public XmlBuilder addChild(String elementName) {
		Element child = this.document.createElement(elementName);
		this.element.appendChild(child);
		return new XmlBuilder(this, this.document, child);
	}
	
	/**
	 * Ends the adding of a child and returns the parent builder
	 * to continue constructing the XML.
	 * @return Parent XmlBuilder
	 */
	public XmlBuilder endChild() {
		if (this.parentBuilder == null) {
			throw new IllegalStateException("Cannot call endChild on root builder");
		}
		
		return this.parentBuilder;
	}

	/**
	 * Gets the document generated by this builder. Usually, you will want to call
	 * this method on the root builder, even though all builders return the same
	 * document.
	 * <p>
	 * This method always returns the same instance of document.
	 * @return Document instance created by this builder
	 */
	public Document getDocument() {
		return this.document;
	}
}
