package ch.inftec.ju.ee.test;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.transaction.UserTransaction;

import org.jboss.logging.Logger;
import org.junit.internal.AssumptionViolatedException;

import ch.inftec.ju.db.JuEmUtil;
import ch.inftec.ju.db.TxHandler;
import ch.inftec.ju.util.JuObjectUtils;
import ch.inftec.ju.util.JuUtils;

/**
 * Helper bean to run container tests in the container.
 * <p>
 * We'll use bean managed transaction management to control the transactions better.
 * @author Martin
 *
 */
@TransactionManagement(TransactionManagementType.BEAN)
public class TestRunnerFacadeBean implements TestRunnerFacade {
	private static Logger logger = Logger.getLogger(TestRunnerFacadeBean.class);
	
	@Inject
	private UserTransaction tx;

	@Inject
	private EntityManager em;
		
//	@Inject
//	private DateProvider dateProvider;
	
	@Override
	public void runPreTestActionsInEjbContext(TestRunnerAnnotationHandler handler) throws Exception {
		try (TxHandler txHandler = new TxHandler(this.tx, true)) { 
			// Execute post test annotations (dataset exporting, data verifying)
			handler.executePreTestAnnotations(new JuEmUtil(this.em));
			txHandler.commit(); // Commit after data verifying / exporting
		}
	}
	
	@Override
	public void runTestMethodInEjbContext(TestRunnerAnnotationHandler handler) throws Exception {
		try (TxHandler txHandler = new TxHandler(this.tx, true)) { 
			logger.debug(String.format("Running Test %s", handler));
	//		this.dateProvider.resetProvider();
			
			// Run the test method
			try {
				handler.executeTestMethod();
			} catch (Throwable t) {
				// Special handling for assumption exceptions. They contain unserializable data, so we create a new
				// one that doesn't contain this extra info.
				if (t instanceof InvocationTargetException) {
					InvocationTargetException ite = (InvocationTargetException) t;
					AssumptionViolatedException ave = JuObjectUtils.as(ite.getTargetException(), AssumptionViolatedException.class);
					if (ave != null) {
						throw new AssumptionViolatedException(ave.getMessage());
					}
				}
				throw t;
			}
			txHandler.commit(); // Perform a commit after the execution of the test method
		}
	}
	
	@Override
	public void runPostTestActionsInEjbContext(TestRunnerAnnotationHandler handler) throws Exception {
		// Run post server annotations in an own annotation so any changed made there is available in the export / verifying phase
		try (TxHandler txHandler = new TxHandler(this.tx, true)) { 
			// Execute post test annotations (dataset exporting, data verifying)
			handler.executePostServerCode(new JuEmUtil(this.em));
			txHandler.commit(); // Commit after data verifying / exporting
		}
		
		// Run post test annotations (export, verify)
		try (TxHandler txHandler = new TxHandler(this.tx, true)) { 
			// Execute post test annotations (dataset exporting, data verifying)
			handler.executePostTestAnnotations(new JuEmUtil(this.em));
			txHandler.commit(); // Commit after data verifying / exporting
		} finally {
			if (JuUtils.getJuPropertyChain().get("ju-testing-ee.clearPropertyChainAfterEachTest", Boolean.class, "false")) {
				JuUtils.clearPropertyChain();
			}
		}
	}


	@Override
	public Object runMethodInEjbContext(String className, String methodName,
			Class<?> argumentTypes[], Object[] args) throws Exception {
		try (TxHandler txHandler = new TxHandler(this.tx, true)) {
			Class<?> clazz = Class.forName(className);
			Object instance = clazz.newInstance();
			
			Method method = clazz.getMethod(methodName, argumentTypes);
			Object res = method.invoke(instance, args);
			
			txHandler.commit();
			return res;
		}
	}
}
