package ch.inftec.ju.ee.test;

import java.io.Serializable;
import java.lang.reflect.Method;

import org.junit.runner.Description;

import ch.inftec.ju.ee.client.ServiceLocatorBuilder;
import ch.inftec.ju.ee.test.TestRunnerFacade.ContextAware;
import ch.inftec.ju.ee.test.TestRunnerFacade.Initializable;
import ch.inftec.ju.ee.test.TestRunnerFacade.TestRunnerContext;
import ch.inftec.ju.testing.db.DataVerifier;
import ch.inftec.ju.testing.db.DbTestAnnotationHandler;

/**
 * Helper class to handle test annotations like @DataSet and @DataVerify.
 * <p>
 * The handler is serializable so it can be used for container tests that run code in different JVMs.
 * <p>
 * When calling the execute... methods, the client is responsible that a valid transaction is present.
 * @author Martin
 *
 */
class TestRunnerAnnotationHandler extends DbTestAnnotationHandler implements Serializable {
	private final TestRunnerContext context;
	
	TestRunnerAnnotationHandler(Method method, Description description, TestRunnerContext context) {
		super(method, description);
		
		this.context = context;
	}
	
	TestRunnerContext getContext() {
		return context;
	}
	
	@Override
	protected String getLocalRoot() {
		return this.getContext().getLocalRoot();
	}
	
	@Override
	protected void initTestClass(Object instance) {
		// Try to set the context (if the class is ContextAware)
		if (ContextAware.class.isAssignableFrom(instance.getClass())) {
			((ContextAware) instance).setContext(this.getContext());
		}
		
		// Try to call the init method (if the class implements Initializable)
		if (Initializable.class.isAssignableFrom(instance.getClass())) {
			((Initializable) instance).init();
		}
	}
	
	@Override
	protected void initVerifier(DataVerifier verifier) {
		if (verifier instanceof DataVerifierCdi) {
			((DataVerifierCdi) verifier).init(ServiceLocatorBuilder.buildLocal().createServiceLocator());
		}
	}
	
	/**
	 * Runs the actual test method.
	 * <p>
	 * This is intended to be used by remote tests that need to run the test method in another VM as the JUnit test runs.
	 * @throws Exception
	 */
	public final void executeTestMethod() throws Exception {
		// Get an instance of the test method so we can invoke it
		Class<?> clazz = Class.forName(this.testClassName);
		Object instance = clazz.newInstance();
		Method method = clazz.getMethod(this.testMethodName);
		
		this.initTestClass(instance);
		
		// Invoke the method
		method.invoke(instance);
	}
}
