package ch.inftec.ju.testing.db;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation to export (and verify) test data after the test has succeeded. When verifying, the resource
 * must be present on the classpath for comparison.
 * <p>
 * The data is exported as soon as the test succeeds, before any verifiers are run.
 * <p>
 * By default, the data set is exported to <code>src/main/resources/dataSetExport</code> to a name
 * like <code>TestClassName_testMethodName.xml</code>. Any leading non-alphabetically characters
 * will be stripped, e.g. _01_test -> test
 * <p>
 * The target location can be modified using the <code>targetDir</code> attribute.
 * <p>
 * No export will be performed if the property <code>ju-testing.export.compareToResource</code> is true.
 * In this case, the export will be done in-memory and compared to a resource on the classpath, using
 * the <code>resourcePrefix</code> attribute to resolve it.
 * @author Martin
 *
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.TYPE, ElementType.METHOD})
public @interface DataSetExport {
	/**
	 * Name of the data set that contains the tables we should export.
	 * @return Path to the export data set XML file resource 
	 */
	String tablesDataSet();
	
	/**
	 * Target directory to copy resources to.
	 * <p>
	 * Defaults to <code>src/main/resources/dataSetExport</code>
	 * @return Data
	 */
	String targetDir() default "src/main/resources/dataSetExport";
	
	/**
	 * Prefix of the resources on the classpath, used to perform.
	 * <p>
	 * Defaults to <code>dataSetExport</code>
	 * @return resource prefix
	 */
	String resourcePrefix() default "dataSetExport";
	
	/**
	 * Flag that indicates if the file should be exported physically. Defaults to true.
	 * <p>
	 * If false, the export is printed to the Logger and can be used as XMLDocument in a
	 * DataVerifier.
	 * <p>
	 * If physical export is disabled, there will be done no resource verification either.
	 * @return Whether export should be physically written to a file
	 */
	boolean doPhysicalExport() default true;
}