package ch.inftec.ju.db;

import java.util.List;

import javax.persistence.EntityManager;


/**
 * Interface providing helper functionality for (raw JDBC) DB connections.
 * <p>
 * Note that implementations will not handle DB transactions - they will just assume that
 * a transaction exists for the wrapped DB connection.
 * @author Martin
 *
 */
public interface JuConnUtil {
	/**
	 * Gets the type of the DB implementation of this EntityManager. If the type is not known (or supported)
	 * by JuEmUtil, an exception is thrown.
	 * @return DbType
	 */
	DbType getDbType();
	
	/**
	 * Executes some DB work using a raw JDBC connection.
	 * @param work DbWork callback interface
	 */
	void doWork(DbWork work);
	
	/**
	 * Executes some DB work using a DataSource.
	 * @param work DsWork callback interface
	 */
	void doWork(DsWork work);
	
	/**
	 * Gets Connection MetaData info
	 * @return MetaDataInfo instance
	 */
	MetaDataInfo getMetaDataInfo();
	
	/**
	 * Gets an instance of a DbHandler to perform DB specific tasks.
	 * @return DbHandler implementation
	 */
	DbHandler getDbHandler();
	
	/**
	 * MetaDataInfo holder
	 * @author Martin
	 *
	 */
	public static interface MetaDataInfo {
		/**
		 * Gets the UserName retrieved from the Connection MetaData
		 * @return Username
		 */
		String getUserName();
		
		/**
		 * Gets the URL retrieved from the Connection MetaData
		 * @return URL
		 */
		String getUrl();
		
		/**
		 * Gets a list of the primary key columns for the specified table
		 * <p>
		 * Column names are kept the way the driver returns them (may be upper, lower or mixed case)
		 * @param tableName Table name
		 * @return List of all columns that make up the primary key. If no primary key is applied, an empty list is returned.
		 */
		List<String> getPrimaryKeyColumns(String tableName);
		
		/**
		 * Gets a list of all table names of the DB. Table names are returned the way the DB driver
		 * returns them, which may be lower, mixed or upper case.
		 * @return List of Table names
		 */
		List<String> getTableNames();
	}
	
	/**
	 * Helper interface to execute DB specific tasks.
	 * @author Martin
	 *
	 */
	public static interface DbHandler {
		/**
		 * Converts the casing of the specified tableName so the DB will understand it.
		 * @param tableName Table name
		 * @return Table name in casing the DB will understand
		 */
		String convertTableNameCasing(String tableName);
		
		/**
		 * Gets a list of all sequence names of the DB, as returned by the driver.
		 * @return List of sequence names
		 */
		List<String> getSequenceNames();
		
		/**
		 * Gets the next value from the specified sequence.
		 * @param sequenceName Sequence name
		 * @return Next value for the sequence
		 */
		Long getNextValueFromSequence(String sequenceName);
		
		/**
		 * Resets identity generation of all tables or sequences to allow for predictable
		 * and repeatable entity generation.
		 * @param val Value for the next primary key
		 */
		void resetIdentityGenerationOrSequences(int val);
	}
	
	/**
	 * Method to extract info from the DatabaseMetaData.
	 * @param action Callback method to do the data extracting into an arbitrary data structure.
	 * @return Data as returned by the callback function
	 */
	public <T> T extractDatabaseMetaData(final DatabaseMetaDataCallback<T> action);
	
	public enum DbType {
		DERBY {
			@Override
			protected DbHandler getDbSpecificHandler(JuConnUtil connUtil) {
				return new DbSpecificHandlerDerby(connUtil);
			}
		},
		H2 {
			@Override
			protected DbHandler getDbSpecificHandler(JuConnUtil connUtil) {
				return new DbSpecificHandlerH2(connUtil);
			}
		},
		MYSQL {
			@Override
			protected DbHandler getDbSpecificHandler(JuConnUtil connUtil) {
				return new DbSpecificHandlerMySql(connUtil);
			}
		},
		ORACLE {
			@Override
			protected DbHandler getDbSpecificHandler(JuConnUtil connUtil) {
				return new DbSpecificHandlerOracle(connUtil);
			}
		};
		
		static DbType evaluateDbType(String productName) {
			if (productName.toLowerCase().contains("derby")) {
				return DERBY;
			} else if (productName.toLowerCase().contains("h2")) {
				return H2; 
			} else if (productName.toLowerCase().contains("mysql")) {
				return MYSQL;
			} else if (productName.toLowerCase().contains("oracle")) {
				return ORACLE;
			} else {
				throw new JuDbException("Unknown DB. Product name: " + productName);
			}
		}
		
		protected abstract DbHandler getDbSpecificHandler(JuConnUtil connUtil);
	}
}
