package ch.vd.shared.iam.web.filter.remote;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.filter.GenericFilterBean;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import java.io.IOException;
import java.util.regex.Pattern;

/**
 * Filtre qui permet de refuser une requête en fonction de l'adresses IP du client qui appelle.
 * <p>
 * Il est possible de spécifier une liste d'adresses IP autorisées à passer le filtre (whitelist) ainsi qu'une liste d'adresses IP interdites (blacklist). La liste
 * des adresses IP interdites a la précédence sur la liste des adresses IP autorisées (en clair, cela veut dire qu'une adresse IP qui apparaîtrait dans les deux
 * listes se verrait refuser le passage par le filtre).
 * <p>
 * Exemples de formats supportés :
 * <pre>
 *     allowed=10.6.3.0
 *     allowed=10.6.3.0,10.6.3.122,10.6.3.220
 *     allowed=10.6.3.*
 *     allowed=10.*
 *     allowed=*
 *     allowed=10\\.120\\.195\\.([2-5][0-9]|6[0-4])
 * </pre>
 * <b>Note :</b> le caractère <code>*</code> permet de remplacer une suite de un ou plusieurs chiffres (ou points).
 */
public class RemoteHostSpringFilter extends GenericFilterBean {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private Pattern[] allowed;
    private Pattern[] denied;

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        String clientAddr = request.getRemoteAddr();

        if (isInvalidAddress(clientAddr)) {
            RemoteHostHelper.handleInvalidAccess(request, response, clientAddr);
            return;
        }

        chain.doFilter(request, response);
    }

    protected boolean isInvalidAddress(String address) {
        return (denied.length > 0 && RemoteHostHelper.hasMatch(address, denied))
                || (allowed.length > 0 && !RemoteHostHelper.hasMatch(address, allowed));
    }

    @Override
    public void destroy() {
        allowed = null;
        denied = null;
    }

    /**
     * @param allowed un pattern pour les adresses IP qui seront autorisées à passer le filtre.
     */
    public void setAllowed(String allowed) {
        logger.debug("allowed='{}'", StringUtils.trimToEmpty(allowed));
        this.allowed = RemoteHostHelper.parseParam(allowed, false);
    }

    /**
     * @param denied un pattern pour les adresses IP qui ne seront pas autorisées à passer le filtre.
     */
    public void setDenied(String denied) {
        logger.debug("denied='{}'", StringUtils.trimToEmpty(denied));
        this.denied = RemoteHostHelper.parseParam(denied, false);
    }

    /**
     * @param allowed un pattern pour les adresses IP qui seront autorisées à passer le filtre.
     */
    public void setAllowedRegex(String allowed) {
        logger.debug("allowed='{}'", StringUtils.trimToEmpty(allowed));
        this.allowed = RemoteHostHelper.parseParam(allowed, true);
    }

    /**
     * @param denied un pattern pour les adresses IP qui ne seront pas autorisées à passer le filtre.
     */
    public void setDeniedRegex(String denied) {
        logger.debug("denied='{}'", StringUtils.trimToEmpty(denied));
        this.denied = RemoteHostHelper.parseParam(denied, true);
    }
}
