package ch.vd.shared.iam.web.filter.auth;

import ch.vd.shared.iam.web.common.IamConstants;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Properties;
import java.util.Vector;


/**
 * Ce filtre est nécessaire aux applications offrant des webservices pour s'assurer que la
 * Session HTTP sera bien fermée à la fin de l'appel.
 * Il ajoute une façade à l'objet requête. Cette facade surcharge les appels aux headers.
 * Ce qui permet de retourner un username, un role et une application lors de l'exécution
 * des filtres IAM qui créent le context de sécurité.
 */
public class PropertiesDevLoginSpringFilter extends AbstractSharedIamAuthenticationFilter {

    private Properties properties;

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        /*
         *  Test si l'application est paramétrer pour contourner l'authentification
         *  - extprop.devlogin.enable=true
         *  Toutefois si les headers IAM_HEADER_* sont fournis par la requête alors la facade
         *  n'est pas appliquée. Cela permet aux tests WIT de simuler l'authentification
         *  d'utilisateurs différents.
         */

        if (request.getHeader(IamConstants.IAM_HEADER_USERNAME) == null
                && request.getHeader(IamConstants.IAM_HEADER_ROLES) == null
                && request.getHeader(IamConstants.IAM_HEADER_APPLICATION) == null) {
            chain.doFilter(new PropertiesDevLoginRequestFacade(request), response);
        } else {
            chain.doFilter(request, response);
        }
    }

    public class PropertiesDevLoginRequestFacade extends
            HttpServletRequestWrapper {

        public PropertiesDevLoginRequestFacade(HttpServletRequest request) {
            super(request);
        }

        @Override
        public String getHeader(String name) {
            switch (name) {
                case IamConstants.IAM_HEADER_USERNAME:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.username"));
                case IamConstants.IAM_HEADER_ROLES:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.roles"));
                case IamConstants.IAM_HEADER_APPLICATION:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.application"));
                case IamConstants.IAM_HEADER_FIRSTNAME:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.first"));
                case IamConstants.IAM_HEADER_LASTNAME:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.last"));
                case IamConstants.IAM_HEADER_AUTHLEVEL:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.authlevel"));
                case IamConstants.IAM_HEADER_EMAIL:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.email"));
                case IamConstants.IAM_HEADER_LASTLOGIN:
                    return StringUtils.stripToNull(properties.getProperty("extprop.devlogin.lastlogin"));
                default:
                    return super.getHeader(name);
            }
        }

        @Override
        @SuppressWarnings({"rawtypes", "unchecked"})
        public Enumeration getHeaders(String name) {

            if (name != null) {
                Vector v = new Vector();
                if (name.equals(IamConstants.IAM_HEADER_USERNAME)) {
                    String str = properties.getProperty("extprop.devlogin.username");
                    v.addElement(str);
                    return v.elements();
                }
                if (name.equals(IamConstants.IAM_HEADER_ROLES)) {
                    String str = properties.getProperty("extprop.devlogin.roles");
                    v.addElement(str);
                    return v.elements();
                }
                if (name.equals(IamConstants.IAM_HEADER_APPLICATION)) {
                    String str = properties.getProperty("extprop.devlogin.application");
                    v.addElement(str);
                    return v.elements();
                }
            }
            return super.getHeaders(name);
        }

        @Override
        @SuppressWarnings({"rawtypes", "unchecked"})
        public Enumeration getHeaderNames() {

            Enumeration e = super.getHeaderNames();
            Vector v = new Vector();
            boolean add_iam_header_username = true;
            boolean add_iam_header_roles = true;
            boolean add_iam_header_application = true;

            while (e.hasMoreElements()) {
                String element = (String) e.nextElement();
                if (element.equals(IamConstants.IAM_HEADER_USERNAME)) {
                    add_iam_header_username = false;
                }
                if (element.equals(IamConstants.IAM_HEADER_ROLES)) {
                    add_iam_header_roles = false;
                }
                if (element.equals(IamConstants.IAM_HEADER_APPLICATION)) {
                    add_iam_header_application = false;
                }
                v.addElement(element);
            }

            if (add_iam_header_username) {
                v.addElement(IamConstants.IAM_HEADER_USERNAME);
            }
            if (add_iam_header_roles) {
                v.addElement(IamConstants.IAM_HEADER_ROLES);
            }
            if (add_iam_header_application) {
                v.addElement(IamConstants.IAM_HEADER_APPLICATION);
            }
            return v.elements();
        }
    }

    public void setProperties(Properties properties) {
        this.properties = properties;
    }
}
