package ch.vd.shared.iam.core.filter.auth;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.mail.internet.MimeUtility;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

public class IamAuthentication extends AbstractIamAuthentication {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final long serialVersionUID = 1L;

    private final String application;
    private final String email;
    private String principal;
    private String username;
    private String firstname;
    private String lastname;
    private final List<String> iamRoles = new ArrayList<>();
    private Date lastLoginTime;

    /**
     * Level d'authentification.
     * Valeurs autorisées:
     * - 10 : password uniquement
     * - 20 : carte Matrix
     */
    private Integer authLevel;

    public IamAuthentication(String application, String principal, String username, String firstName, String lastName, String email,
                             List<String> iamRoles, Integer authLevel, Date lastLoginTime, Collection<IamAuthority> allRoles) {
        super(allRoles);

        init(principal, username, iamRoles, authLevel, lastLoginTime);

        this.application = application;
        try {
            logger.debug("firstName : {}", firstName);
            logger.debug("lastName : {}", lastName);
            firstname = MimeUtility.decodeText(StringUtils.stripToEmpty(firstName));
            lastname = MimeUtility.decodeText(StringUtils.stripToEmpty(lastName));
        } catch (UnsupportedEncodingException e) {
            logger.info("An error occured during MIME headers decoding. Fallback to old assignation for first name and last name");
            firstname = firstName;
            lastname = lastName;
        }
        this.email = email;
    }

    private void init(String principal, String username, List<String> roles, Integer authLevel, Date lastLoginTime) {
        this.principal = principal;
        this.username = username;

        this.authLevel = authLevel;
        this.lastLoginTime = lastLoginTime;

        if (roles != null) {
            iamRoles.addAll(roles);
        }
        setAuthenticated(true);
    }

    @Override
    public Object getCredentials() {
        // Il n'y a pas de credentials (password) avec IAM
        return "";
    }

    @Override
    public String getApplication() {
        return application;
    }

    @Override
    public Object getPrincipal() {
        return principal;
    }

    @Override
    public String getIUP() {
        return getUsername();
    }

    public String getUsername() {
        return username;
    }

    /**
     * @return The user firstname
     */
    @Override
    public String getFirstname() {
        return firstname;
    }

    /**
     * @return The user lastname
     */
    @Override
    public String getLastname() {
        return lastname;
    }

    @Override
    public String getEmail() {
        return email;
    }

    @Override
    public Integer getAuthLevel() {
        return authLevel;
    }

    @Override
    public Date getLastLoginTime() {
        return lastLoginTime;
    }

    /**
     * Ne renvoie que les roles tels que définis dans IAM, rien de plus
     *
     */
    @Override
    public List<String> getIamOriginalRoles() {
        return iamRoles;
    }

    /**
     * Renvoie tous les roles y compris les sub-roles et ceux qui viennent de la DB
     *
     */
    @Override
    public Collection<IamAuthority> getAllRoles() {
        return (Collection) getAuthorities();
    }

    /**
     * @return true si authLevel == 20
     */
    @Override
    public boolean isStrongAuth() {
        return authLevel != null && (authLevel == 20);
    }

    @Override
    public String toString() {
        return "name=" + getPrincipal() + " firstName=" + getFirstname() + " lastName=" + getLastname() + " email=" + getEmail() + " iam-roles=" + getIamOriginalRoles();
    }
}
