/*
 * Decompiled with CFR 0.152.
 */
package ch.dissem.bitmessage.networking;

import ch.dissem.bitmessage.InternalContext;
import ch.dissem.bitmessage.entity.Addr;
import ch.dissem.bitmessage.entity.CustomMessage;
import ch.dissem.bitmessage.entity.GetData;
import ch.dissem.bitmessage.entity.Inv;
import ch.dissem.bitmessage.entity.MessagePayload;
import ch.dissem.bitmessage.entity.NetworkMessage;
import ch.dissem.bitmessage.entity.ObjectMessage;
import ch.dissem.bitmessage.entity.VerAck;
import ch.dissem.bitmessage.entity.Version;
import ch.dissem.bitmessage.entity.valueobject.InventoryVector;
import ch.dissem.bitmessage.entity.valueobject.NetworkAddress;
import ch.dissem.bitmessage.exception.InsufficientProofOfWorkException;
import ch.dissem.bitmessage.exception.NodeException;
import ch.dissem.bitmessage.ports.NetworkHandler;
import ch.dissem.bitmessage.utils.Singleton;
import ch.dissem.bitmessage.utils.UnixTime;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractConnection {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractConnection.class);
    protected final InternalContext ctx;
    protected final Mode mode;
    protected final NetworkAddress host;
    protected final NetworkAddress node;
    protected final NetworkHandler.MessageListener listener;
    protected final Map<InventoryVector, Long> ivCache;
    protected final Deque<MessagePayload> sendingQueue;
    protected final Set<InventoryVector> commonRequestedObjects;
    protected final Set<InventoryVector> requestedObjects;
    protected volatile State state;
    protected long lastObjectTime;
    private final long syncTimeout;
    private long syncReadTimeout = Long.MAX_VALUE;
    protected long peerNonce;
    protected int version;
    protected long[] streams;
    private boolean verackSent;
    private boolean verackReceived;

    public AbstractConnection(InternalContext context, Mode mode, NetworkAddress node, Set<InventoryVector> commonRequestedObjects, long syncTimeout) {
        this.ctx = context;
        this.mode = mode;
        this.host = new NetworkAddress.Builder().ipv6(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0).port(0).build();
        this.node = node;
        this.listener = context.getNetworkListener();
        this.syncTimeout = syncTimeout > 0L ? UnixTime.now((long)syncTimeout) : 0L;
        this.requestedObjects = Collections.newSetFromMap(new ConcurrentHashMap(10000));
        this.ivCache = new ConcurrentHashMap<InventoryVector, Long>();
        this.sendingQueue = new ConcurrentLinkedDeque<MessagePayload>();
        this.state = State.CONNECTING;
        this.commonRequestedObjects = commonRequestedObjects;
    }

    public Mode getMode() {
        return this.mode;
    }

    public NetworkAddress getNode() {
        return this.node;
    }

    public State getState() {
        return this.state;
    }

    public long[] getStreams() {
        return this.streams;
    }

    protected void handleMessage(MessagePayload payload) {
        switch (this.state) {
            case ACTIVE: {
                this.receiveMessage(payload);
                break;
            }
            case DISCONNECTED: {
                break;
            }
            default: {
                this.handleCommand(payload);
            }
        }
    }

    private void receiveMessage(MessagePayload messagePayload) {
        switch (messagePayload.getCommand()) {
            case INV: {
                this.receiveMessage((Inv)messagePayload);
                break;
            }
            case GETDATA: {
                this.receiveMessage((GetData)messagePayload);
                break;
            }
            case OBJECT: {
                this.receiveMessage((ObjectMessage)messagePayload);
                break;
            }
            case ADDR: {
                this.receiveMessage((Addr)messagePayload);
                break;
            }
            default: {
                throw new IllegalStateException("Unexpectedly received '" + messagePayload.getCommand() + "' command");
            }
        }
    }

    private void receiveMessage(Inv inv) {
        int originalSize = inv.getInventory().size();
        this.updateIvCache(inv.getInventory());
        List missing = this.ctx.getInventory().getMissing(inv.getInventory(), this.streams);
        missing.removeAll(this.commonRequestedObjects);
        LOG.trace("Received inventory with " + originalSize + " elements, of which are " + missing.size() + " missing.");
        this.send((MessagePayload)new GetData.Builder().inventory(missing).build());
    }

    private void receiveMessage(GetData getData) {
        for (InventoryVector iv : getData.getInventory()) {
            ObjectMessage om = this.ctx.getInventory().getObject(iv);
            if (om == null) continue;
            this.sendingQueue.offer((MessagePayload)om);
        }
    }

    private void receiveMessage(ObjectMessage objectMessage) {
        this.requestedObjects.remove(objectMessage.getInventoryVector());
        if (this.ctx.getInventory().contains(objectMessage)) {
            LOG.trace("Received object " + objectMessage.getInventoryVector() + " - already in inventory");
            return;
        }
        try {
            this.listener.receive(objectMessage);
            Singleton.cryptography().checkProofOfWork(objectMessage, 1000L, 1000L);
            this.ctx.getInventory().storeObject(objectMessage);
            this.ctx.getNetworkHandler().offer(objectMessage.getInventoryVector());
            this.lastObjectTime = UnixTime.now();
        }
        catch (InsufficientProofOfWorkException e) {
            LOG.warn(e.getMessage());
        }
        catch (IOException e) {
            LOG.error("Stream " + objectMessage.getStream() + ", object type " + objectMessage.getType() + ": " + e.getMessage(), (Throwable)e);
        }
        finally {
            if (!this.commonRequestedObjects.remove(objectMessage.getInventoryVector())) {
                LOG.debug("Received object that wasn't requested.");
            }
        }
    }

    private void receiveMessage(Addr addr) {
        LOG.trace("Received " + addr.getAddresses().size() + " addresses.");
        this.ctx.getNodeRegistry().offerAddresses(addr.getAddresses());
    }

    private void updateIvCache(List<InventoryVector> inventory) {
        this.cleanupIvCache();
        Long now = UnixTime.now();
        for (InventoryVector iv : inventory) {
            this.ivCache.put(iv, now);
        }
    }

    public void offer(InventoryVector iv) {
        this.sendingQueue.offer((MessagePayload)new Inv.Builder().addInventoryVector(iv).build());
        this.updateIvCache(Collections.singletonList(iv));
    }

    public boolean knowsOf(InventoryVector iv) {
        return this.ivCache.containsKey(iv);
    }

    private void cleanupIvCache() {
        Long fiveMinutesAgo = UnixTime.now((long)-300L);
        for (Map.Entry<InventoryVector, Long> entry : this.ivCache.entrySet()) {
            if (entry.getValue() >= fiveMinutesAgo) continue;
            this.ivCache.remove(entry.getKey());
        }
    }

    private void handleCommand(MessagePayload payload) {
        switch (payload.getCommand()) {
            case VERSION: {
                this.handleVersion((Version)payload);
                break;
            }
            case VERACK: {
                if (this.verackSent) {
                    this.activateConnection();
                }
                this.verackReceived = true;
                break;
            }
            case CUSTOM: {
                MessagePayload response = this.ctx.getCustomCommandHandler().handle((CustomMessage)payload);
                if (response == null) {
                    this.disconnect();
                    break;
                }
                this.send(response);
                break;
            }
            default: {
                throw new NodeException("Command 'version' or 'verack' expected, but was '" + payload.getCommand() + "'");
            }
        }
    }

    private void activateConnection() {
        LOG.info("Successfully established connection with node " + this.node);
        this.state = State.ACTIVE;
        this.node.setTime(UnixTime.now());
        if (this.mode != Mode.SYNC) {
            this.sendAddresses();
            this.ctx.getNodeRegistry().offerAddresses(Collections.singletonList(this.node));
        }
        this.sendInventory();
    }

    private void sendAddresses() {
        List addresses = this.ctx.getNodeRegistry().getKnownAddresses(1000, this.streams);
        this.sendingQueue.offer((MessagePayload)new Addr.Builder().addresses((Collection)addresses).build());
    }

    private void sendInventory() {
        List inventory = this.ctx.getInventory().getInventory(this.streams);
        for (int i = 0; i < inventory.size(); i += 50000) {
            this.sendingQueue.offer((MessagePayload)new Inv.Builder().inventory(inventory.subList(i, Math.min(inventory.size(), i + 50000))).build());
        }
    }

    private void handleVersion(Version version) {
        if (version.getNonce() == this.ctx.getClientNonce()) {
            LOG.info("Tried to connect to self, disconnecting.");
            this.disconnect();
        } else if (version.getVersion() >= 3) {
            this.peerNonce = version.getNonce();
            if (this.peerNonce == this.ctx.getClientNonce()) {
                this.disconnect();
            }
            this.version = version.getVersion();
            this.streams = version.getStreams();
            this.verackSent = true;
            this.send((MessagePayload)new VerAck());
            if (this.mode == Mode.SERVER) {
                this.send((MessagePayload)new Version.Builder().defaults(this.ctx.getClientNonce()).addrFrom(this.host).addrRecv(this.node).build());
            }
            if (this.verackReceived) {
                this.activateConnection();
            }
        } else {
            LOG.info("Received unsupported version " + version.getVersion() + ", disconnecting.");
            this.disconnect();
        }
    }

    protected boolean syncFinished(NetworkMessage msg) {
        if (this.mode != Mode.SYNC) {
            return false;
        }
        if (Thread.interrupted()) {
            return true;
        }
        if (this.state != State.ACTIVE) {
            return false;
        }
        if (this.syncTimeout < UnixTime.now()) {
            LOG.info("Synchronization timed out");
            return true;
        }
        if (!this.sendingQueue.isEmpty()) {
            this.syncReadTimeout = System.currentTimeMillis() + 1000L;
            return false;
        }
        if (msg == null) {
            return this.syncReadTimeout < System.currentTimeMillis();
        }
        this.syncReadTimeout = System.currentTimeMillis() + 1000L;
        return false;
    }

    public void disconnect() {
        this.state = State.DISCONNECTED;
        this.ctx.getNetworkHandler().request(this.requestedObjects);
    }

    protected abstract void send(MessagePayload var1);

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        AbstractConnection that = (AbstractConnection)o;
        return Objects.equals(this.node, that.node);
    }

    public int hashCode() {
        return Objects.hash(this.node);
    }

    public static enum State {
        CONNECTING,
        ACTIVE,
        DISCONNECTED;

    }

    public static enum Mode {
        SERVER,
        CLIENT,
        SYNC;

    }
}

