/*-
 * #%L
 * QR Invoice Solutions
 * %%
 * Copyright (C) 2017 - 2019 Codeblock GmbH
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * -
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * -
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 * -
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses are available for this software. These replace the above 
 * AGPLv3 terms and offer support, maintenance and allow the use in commercial /
 * proprietary products.
 * -
 * More information on commercial licenses are available at the following page:
 * https://www.qr-invoice.ch/licenses/
 * #L%
 */
package ch.codeblock.qrinvoice.tools.generator;

import ch.codeblock.qrinvoice.FontFamily;
import ch.codeblock.qrinvoice.OutputResolution;
import ch.codeblock.qrinvoice.PageSize;
import ch.codeblock.qrinvoice.model.AdditionalInformation;
import ch.codeblock.qrinvoice.model.AddressType;
import ch.codeblock.qrinvoice.model.AlternativeSchemes;
import ch.codeblock.qrinvoice.model.Creditor;
import ch.codeblock.qrinvoice.model.CreditorInformation;
import ch.codeblock.qrinvoice.model.Header;
import ch.codeblock.qrinvoice.model.PaymentAmountInformation;
import ch.codeblock.qrinvoice.model.PaymentReference;
import ch.codeblock.qrinvoice.model.QrInvoice;
import ch.codeblock.qrinvoice.model.ReferenceType;
import ch.codeblock.qrinvoice.model.SwissPaymentsCode;
import ch.codeblock.qrinvoice.model.UltimateCreditor;
import ch.codeblock.qrinvoice.model.UltimateDebtor;
import ch.codeblock.qrinvoice.model.annotation.Description;
import ch.codeblock.qrinvoice.model.annotation.Example;
import ch.codeblock.qrinvoice.model.annotation.Mandatory;
import ch.codeblock.qrinvoice.model.annotation.Size;
import ch.codeblock.qrinvoice.paymentpartreceipt.LayoutDefinitions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.reflect.FieldUtils;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Currency;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class RestModelGenerator {
    private static final String INDENT = "    "; // == one tab

    private static final String REST_MODEL_PACKAGE = "ch.codeblock.qrinvoice.rest.model";
    private static final Map<String, String> REPLACEMENT_MAP = new HashMap<>();
    static {
        REPLACEMENT_MAP.put("ch.codeblock.qrinvoice.model", REST_MODEL_PACKAGE);
        REPLACEMENT_MAP.put("java.util.", REST_MODEL_PACKAGE + ".");
        REPLACEMENT_MAP.put(PageSize.class.getName(), REST_MODEL_PACKAGE + "." + PageSize.class.getSimpleName());
        REPLACEMENT_MAP.put("OutputResolution", "OutputResolutionEnum");
        REPLACEMENT_MAP.put("AddressType", "AddressTypeEnum");
        REPLACEMENT_MAP.put("ReferenceType", "ReferenceTypeEnum");
        REPLACEMENT_MAP.put("Currency", "CurrencyEnum");
        REPLACEMENT_MAP.put("PageSize", "PageSizeEnum");
        REPLACEMENT_MAP.put("FontFamily", "FontFamilyEnum");
        REPLACEMENT_MAP.put("Locale", "LanguageEnum");
    }
    private static final Map<String, Set<String>> IGNORE_MAP = new HashMap<>();
    static {
        IGNORE_MAP.put("AdditionalInformation", new HashSet<>(Collections.singletonList("trailer")));
    }

    private static Path packageDir;

    public static void main(String[] args) throws IOException {
        final Path outputModule = Paths.get(args[0]);
        if (!Files.exists(outputModule)) {
            throw new RuntimeException("Output Module does not exist");
        }

        packageDir = Files.createDirectories(outputModule.resolve("src/main/java").resolve(REST_MODEL_PACKAGE.replace('.', '/')));

        generate(Currency.class);
        generate(Locale.class);
        generate(PageSize.class);
        generate(FontFamily.class);
        generate(OutputResolution.class);

        // Header is not exposed
        // generate(Header.class);

        generate(AddressType.class);
        generate(AlternativeSchemes.class);
        generate(Creditor.class);
        generate(CreditorInformation.class);
        generate(PaymentAmountInformation.class);
        generate(PaymentReference.class);
        generate(ReferenceType.class);
        generate(AdditionalInformation.class);
        generate(QrInvoice.class);
        generate(UltimateDebtor.class);
        generate(UltimateCreditor.class);
    }

    public static void generate(final Class<?> clazz) throws IOException {
        final Path javaFile = packageDir.resolve(transformType(clazz.getSimpleName(), null) + ".java");
        System.out.println(javaFile.toFile().getAbsolutePath());
        try (PrintWriter printWriter = new PrintWriter(new FileOutputStream(javaFile.toFile()))) {
            if (clazz.isEnum()) {
                enumeration(clazz, printWriter);
            } else if (clazz.equals(Currency.class)) {
                currencyEnum(clazz, printWriter);
            } else if (clazz.equals(Locale.class)) {
                localeEnum(clazz, printWriter);
            } else {
                clazz(clazz, printWriter);
            }
        }
    }

    private static void currencyEnum(final Class<?> clazz, final PrintWriter printWriter) {
        final Stream<String> valuesStream = SwissPaymentsCode.SUPPORTED_CURRENCIES.stream().map(Currency::getCurrencyCode);
        enumeration(clazz, printWriter, valuesStream);
    }


    private static void localeEnum(final Class<?> clazz, final PrintWriter printWriter) {
        final Stream<String> valuesStream = LayoutDefinitions.SUPPORTED_LOCALES.stream().map(Locale::getLanguage);
        enumeration(clazz, printWriter, valuesStream);
    }

    private static void enumeration(final Class<?> clazz, final PrintWriter printWriter, final Stream<String> valuesStream) {
        printWriter.printf("package %s;\n", REST_MODEL_PACKAGE);
        printWriter.println();
        printWriter.printf("public enum %s {", transformType(clazz.getSimpleName(), null));
        printWriter.print(valuesStream.map(v -> System.lineSeparator() + INDENT + v).collect(Collectors.joining(",")));
        printWriter.println(";");
        printWriter.println();
        printWriter.println(INDENT + "@Override");
        printWriter.println(INDENT + "@com.fasterxml.jackson.annotation.JsonValue");
        printWriter.println(INDENT + "public String toString() {");
        printWriter.println(INDENT + "    return name();");
        printWriter.println(INDENT + "}");
        printWriter.println();
        printWriter.println(INDENT + "@com.fasterxml.jackson.annotation.JsonCreator");
        printWriter.printf("    public static %s fromValue(String text) {\n", transformType(clazz.getSimpleName(), null));
        printWriter.printf("        for (%s v : %s.values()) {\n", transformType(clazz.getSimpleName(), null), transformType(clazz.getSimpleName(), null));
        printWriter.println(INDENT + "        if (String.valueOf(v.name()).equals(text)) {");
        printWriter.println(INDENT + "            return v;");
        printWriter.println(INDENT + "        }");
        printWriter.println(INDENT + "    }");
        printWriter.println(INDENT + "    return null;");
        printWriter.println(INDENT + "}");
        printWriter.println("}");
    }

    private static void enumeration(final Class<?> clazz, final PrintWriter printWriter) {
        printWriter.printf("package %s;\n", REST_MODEL_PACKAGE);
        printWriter.println();
        printWriter.printf("@javax.annotation.Generated(value = \"ch.codeblock.qrinvoice.tools.ch.codeblock.qrinvoice.tools.generator.RestModelGenerator\", date = \"%s\")\n", LocalDateTime.now().toString());

        final String newEnumClassName = transformType(clazz.getSimpleName(), clazz.getName());
        printWriter.println("public enum " + newEnumClassName + " {");

        for (int i = 0; i < clazz.getEnumConstants().length; i++) {
            final Object o = clazz.getEnumConstants()[i];
            final String enumValuePostfix = i == clazz.getEnumConstants().length - 1 ? ";" : ",";
            if (o instanceof ReferenceType) {
                final ReferenceType referenceType = (ReferenceType) o;
                printWriter.printf(INDENT + "%s(\"%s\")%s\n", referenceType.name(), referenceType.getReferenceTypeCode(), enumValuePostfix);
            } else if (o instanceof Enum) {
                final Enum e = (Enum) o;
                printWriter.printf(INDENT + "%s(\"%s\")%s\n", e.name(), e.name(), enumValuePostfix);
            } else {
                throw new RuntimeException();
            }
        }

        printWriter.println();
        printWriter.println(INDENT + "private final String code;");
        printWriter.println();
        printWriter.printf(INDENT + "%s(final String code) {\n", newEnumClassName);
        printWriter.println(INDENT + INDENT + "this.code = code;");
        printWriter.println(INDENT + "}");
        printWriter.println();
        printWriter.println(INDENT + "public String getCode() {");
        printWriter.println(INDENT + INDENT + "return code;");
        printWriter.println(INDENT + "}");
        printWriter.println();
        printWriter.println(INDENT + "@Override");
        printWriter.println(INDENT + "@com.fasterxml.jackson.annotation.JsonValue");
        printWriter.println(INDENT + "public String toString() {");
        printWriter.println(INDENT + "    return String.valueOf(code);");
        printWriter.println(INDENT + "}");
        printWriter.println();
        printWriter.println(INDENT + "@com.fasterxml.jackson.annotation.JsonCreator");
        printWriter.printf(INDENT + "public static %s fromValue(String text) {\n", newEnumClassName);
        printWriter.printf(INDENT + "    for (%s b : %s.values()) {\n", newEnumClassName, newEnumClassName);
        printWriter.println(INDENT + "        if (String.valueOf(b.code).equals(text)) {");
        printWriter.println(INDENT + "            return b;");
        printWriter.println(INDENT + "        }");
        printWriter.println(INDENT + "    }");
        printWriter.println(INDENT + "    return null;");
        printWriter.println(INDENT + "}");

        printWriter.println("}");
    }

    private static void clazz(final Class<?> clazz, final PrintWriter printWriter) {
        printWriter.printf("package %s;\n", REST_MODEL_PACKAGE);
        printWriter.println();
        printWriter.printf("@javax.annotation.Generated(value = \"ch.codeblock.qrinvoice.tools.ch.codeblock.qrinvoice.tools.generator.RestModelGenerator\", date = \"%s\")\n", LocalDateTime.now().toString());

        printWriter.println("public class " + clazz.getSimpleName() + " {");
        final List<Field> filteredFieldsList = FieldUtils.getAllFieldsList(clazz).stream()
                .filter(field -> !field.getType().equals(Header.class))
                .filter(field -> !(IGNORE_MAP.containsKey(clazz.getSimpleName()) && IGNORE_MAP.get(clazz.getSimpleName()).contains(field.getName())))
                .collect(Collectors.toList());
        for (final Field field : filteredFieldsList) {
            printWriter.println();

            final Class<?> type = field.getType();
            final String name = field.getName();
            final int modifiers = field.getModifiers();

            final Optional<Method> optionalGetter = getGetter(clazz, field);
            if (optionalGetter.isPresent()) {
                final Method getterMethod = optionalGetter.get();
                final boolean required = getterMethod.getAnnotation(Mandatory.class) != null;

                final String example = getGetterExampleString(getterMethod);
                final String description = getGetterDescription(getterMethod);

                final Size sizeAnnotation = getterMethod.getAnnotation(Size.class);

                if (sizeAnnotation != null) {
                    printWriter.printf(INDENT + "@javax.validation.constraints.Size(min = %s, max = %s)\n", sizeAnnotation.min(), sizeAnnotation.max());
                }
                if (required) {
                    printWriter.println(INDENT + "@javax.validation.constraints.NotNull");
                }
                printWriter.printf(INDENT + "@io.swagger.annotations.ApiModelProperty(required = %s, value = \"%s\", notes = \"%s\", example = \"%s\")\n", required, description, description, example);

            }

            printWriter.printf(INDENT + "@com.fasterxml.jackson.annotation.JsonProperty(\"%s\")\n", field.getName());

            final String visibilityModifier = getVisibilityModifierString(modifiers);
            printWriter.printf(INDENT + "%s %s %s;\n", visibilityModifier, transformType(type.getName(), name), name);
        }

        printWriter.println();
        for (final Field field : filteredFieldsList) {
            final Optional<Method> optionalGetter = getGetter(clazz, field);
            if (optionalGetter.isPresent()) {
                final Method getterMethod = optionalGetter.get();

                printWriter.printf(INDENT + "%s %s %s(%s) {\n", "public" /*TODO*/, transformType(getterMethod.getReturnType().getName(), field.getName()), getterMethod.getName(), "");
                printWriter.printf(INDENT + INDENT + "return this.%s;\n", field.getName());
                printWriter.println(INDENT + "}");
            }

            printWriter.println();

            String setter = "set" + StringUtils.capitalize(field.getName());
            try {
                final Method setterMethod = clazz.getDeclaredMethod(setter, field.getType());
                final StringBuilder params = new StringBuilder();
                boolean first = true;
                for (final Parameter parameter : setterMethod.getParameters()) {
                    if (!first) {
                        params.append(", ");
                    }
                    params.append(transformType(parameter.getType().getName(), field.getName()));
                    params.append(" ");
                    params.append(field.getName()); // parameter.getName()
                    first = false;
                }
                printWriter.printf(INDENT + "%s void %s(%s) {\n", "public" /*TODO*/, setterMethod.getName(), params.toString());
                printWriter.printf(INDENT + INDENT + "this.%s = %s;\n", field.getName(), field.getName());
                printWriter.println(INDENT + "}");
            } catch (NoSuchMethodException e) {
                System.err.println(e.getMessage());
            }

            printWriter.println();
        }

        printWriter.println("}");
    }

    private static String getGetterExampleString(final Method getterMethod) {
        final Example exampleAnnotation = getterMethod.getAnnotation(Example.class);
        return exampleAnnotation != null ? exampleAnnotation.value() : "";
    }

    private static String getGetterDescription(final Method getterMethod) {
        final Description descriptionAnnotation = getterMethod.getAnnotation(Description.class);
        final String rawDescription = (descriptionAnnotation != null ? descriptionAnnotation.value() : "");
        return rawDescription.replaceAll("\"", "\\\\\"");
    }

    private static String getVisibilityModifierString(final int modifiers) {
        final String visibilityModifier;
        if (Modifier.isPrivate(modifiers)) {
            visibilityModifier = "private";
        } else if (Modifier.isProtected(modifiers)) {
            visibilityModifier = "protected";
        } else if (Modifier.isPublic(modifiers)) {
            visibilityModifier = "public";
        } else {
            visibilityModifier = "";
        }
        return visibilityModifier;
    }

    private static Optional<Method> getGetter(final Class<?> clazz, final Field field) {
        String getter = "get" + StringUtils.capitalize(field.getName());
        return Stream.of(clazz.getDeclaredMethods()).filter(m -> m.getName().equals(getter)).findFirst();
    }

    private static String transformType(final String type, final String fieldName) {
        return transformTypeNames(replaceType(type, fieldName));
    }


    private static String transformTypeNames(String type) {
        for (final Map.Entry<String, String> entry : REPLACEMENT_MAP.entrySet()) {
            type = type.replace(entry.getKey(), entry.getValue());
        }
        return type;
    }

    private static String replaceType(final String type, final String fieldName) {
        if ("alternativeSchemeParameters".equalsIgnoreCase(fieldName) && List.class.getName().equals(type)) {
            return "String[]";
        } else {
            return type;
        }
    }
}
