package ch.bullfin.httpmanager.request_queue;

import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.AsyncTask;
import android.util.Log;

import ch.bullfin.httpmanager.HTTPManager;
import ch.bullfin.httpmanager.Response;

/**
 * @author Rahul Raveendran V P
 *         Created on 26/2/15 @ 7:33 PM
 *         https://github.com/rahulrvp
 */
public class RequestQueue {
    private static final String LOG_TAG = RequestQueue.class.getSimpleName();
    private static final Boolean lock = true;

    private static final long RETRY_INTERVAL = 5 * 60 * 1000;

    private static Status mStatus;
    private static Context mContext;
    private static RequestQueue mInstance = null;

    private RequestQueue() {
        mStatus = Status.AVAILABLE;
    }

    public synchronized static RequestQueue getInstance(Context context) {
        mContext = context;

        if (mInstance == null) {
            mInstance = new RequestQueue();
        }

        return mInstance;
    }

    public long enqueue(Request request) {
        Log.d(LOG_TAG, "Adding new request to the Queue.");
        if (request != null) {
            synchronized (lock) {
                RequestQueueDB db = new RequestQueueDB(mContext);
                db.insert(request);
                db.close();
            }

            sync();
            return request.getId();
        }

        return -1;
    }

    private void remove(Request request) {
        if (request != null) {
            Log.d(LOG_TAG, "Removing request with id:" + request.getId() + " from the Queue");
            synchronized (lock) {
                RequestQueueDB db = new RequestQueueDB(mContext);
                db.delete(request);
                db.close();
            }
        }
    }

    private Request fetchRequest() {
        synchronized (lock) {
            if (mStatus == Status.AVAILABLE && isConnectedToNetwork()) {
                mStatus = Status.BUSY;
                Request request = null;
                RequestQueueDB db = new RequestQueueDB(mContext);
                Object reqObject = db.getNextRequest();
                if (reqObject != null) {
                    request = (Request) reqObject;
                }
                db.close();

                if (request == null) {
                    mStatus = Status.AVAILABLE;
                }
                return request;
            }
        }

        return null;
    }

    public void sync() {
        Request request = fetchRequest();

        if (request != null) {
            if (request.getPendingRetryCount() != 0) {
                long nextRetryTime = request.getNextRetryTime();
                // Checks if next retry time had passed or not. (0 is the default time for normal requests.)
                if (nextRetryTime == 0 || nextRetryTime < System.currentTimeMillis()) {
                    new SendRequestTask(request).execute();
                } else {
                    // The request has a next retry time which had not passed yet, so we need to move that to back of the queue.
                    remove(request);
                    synchronized (lock) {
                        mStatus = Status.AVAILABLE;
                    }
                    enqueue(request);
                }
            } else {
                // Request expired its retries. Need to be removed from queue. This is an escape plan. This case will not happen normally.
                remove(request);
                synchronized (lock) {
                    mStatus = Status.AVAILABLE;
                }
            }
        }
    }

    private boolean isConnectedToNetwork() {
        ConnectivityManager connectivityManager =
                (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);

        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();

        return activeNetworkInfo != null && activeNetworkInfo.isConnectedOrConnecting();
    }

    private enum Status {
        AVAILABLE,
        BUSY
    }

    private class SendRequestTask extends AsyncTask<Void, Void, Response> {
        private Request mRequest;

        public SendRequestTask(Request request) {
            mRequest = request;
        }

        @Override
        protected void onPreExecute() {
            mStatus = RequestQueue.Status.BUSY;
        }

        @Override
        protected Response doInBackground(Void... params) {
            Response response = null;

            if (mRequest != null) {
                Log.d(LOG_TAG, "Processing request (id: " + mRequest.getId() + ").");
                HTTPManager httpManager = new HTTPManager(mRequest.getUrl());

                if (mRequest.isMultipart()) {
                    switch (mRequest.getMethod()) {
                        case POST:
                            response = httpManager.postMultiPart(mRequest.getParts());
                            break;

                        case PUT:
                            response = httpManager.putMultiPart(mRequest.getParts());
                            break;

                        default:
                            Log.e(LOG_TAG, "Unknown method" + mRequest.getMethod());
                    }
                } else {
                    switch (mRequest.getMethod()) {
                        case GET:
                            response = httpManager.get(mRequest.getUrlParams());
                            break;

                        case POST:
                            response = httpManager.post(mRequest.getParamJson());
                            break;

                        case PUT:
                            response = httpManager.put(mRequest.getParamJson());
                            break;

                        default:
                            Log.e(LOG_TAG, "Unknown method" + mRequest.getMethod());
                    }
                }
            }

            return response;
        }

        @Override
        protected void onPostExecute(Response response) {
            remove(mRequest);

            if (response.getStatusCode() / 100 == 2) {
                Log.d(LOG_TAG, "Request success for request (id: " + mRequest.getId() + "). Status code: " + response.getStatusCode());

                Intent intent = new Intent("ch.bullfin.httpmanager.send_complete");
                intent.putExtra("request_id", mRequest.getId());
                intent.putExtra("response_status", response.getStatusCode());
                intent.putExtra("response_body", response.getResponseBody());
                mContext.sendBroadcast(intent);
            } else {
                Log.d(LOG_TAG, "Request failed for request (id: " + mRequest.getId() + "). Status code: " + response.getStatusCode() + ". Response: " + response.getResponseBody());
                int retryCount = mRequest.getPendingRetryCount() - 1;
                if (retryCount > 0) {
                    mRequest.setPendingRetryCount(retryCount);
                    // setting a retry time 5 minutes from now.
                    long nextRetryTime = (RETRY_INTERVAL) + System.currentTimeMillis();
                    mRequest.setNextRetryTime(nextRetryTime);

                    enqueue(mRequest);
                } else {
                    Log.d(LOG_TAG, "Request failed for request (id: " + mRequest.getId() + "). Status code: " + response.getStatusCode());

                    Intent intent = new Intent("ch.bullfin.httpmanager.send_failed");
                    intent.putExtra("request_id", mRequest.getId());
                    intent.putExtra("response_status", response.getStatusCode());
                    intent.putExtra("response_body", response.getResponseBody());
                    mContext.sendBroadcast(intent);
                }
            }

            mStatus = RequestQueue.Status.AVAILABLE;
            sync();
        }
    }
}
