package ch.bullfin.httpmanager;

import android.content.Context;
import android.util.Log;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.X509TrustManager;

import ch.bullfin.httpmanager.request_queue.Request;
import ch.bullfin.httpmanager.request_queue.RequestQueue;


/**
 * Abstraction for TCP/IP communication.
 * <p>
 * Author : Asif CH (asif@bullfin.ch)
 * Bullfinch Software Pvt Ltd
 */

public class HTTPManager {
    private static final String LOG_TAG = "HTTPManager";

    private String mUrl;
    private HttpURLConnection mUrlConnection;
    private HashMap<String, String> mRequestProperties;
    private boolean mIgnoreHostnameVerification;
    private String mAuthToken;
    private Context mContext;

    public static final int HTTP_OK = 200;
    public static final int HTTP_FILE_NOT_FOUND = 404;
    public static final int HTTP_SERVER_ERROR = 500;
    private static final String boundary = "000boundary000";

    public HTTPManager(String url) {
        mUrl = url;
        mRequestProperties = new HashMap<String, String>();
        mIgnoreHostnameVerification = false;
        mAuthToken = null;

        addRequestProperty("User-Agent", "Mozilla/five.0 ( compatible ) ");
        addRequestProperty("Accept", "application/json");
        addRequestProperty("Connection", "close");
    }

    public HTTPManager setContext(Context context) {
        this.mContext = context;
        return this;
    }

    public HTTPManager addRequestProperty(String key, String value) {
        mRequestProperties.put(key, value);
        return this;
    }

    public HTTPManager ignoreHostnameVerification() {
        mIgnoreHostnameVerification = true;
        return this;
    }

    public HTTPManager setCookie(String cookie) {
        addRequestProperty("Cookie", cookie);
        return this;
    }

    public HTTPManager setAuthToken(String authToken) {
        mAuthToken = authToken;
        return this;
    }

    private void prepareRequest() {
        URL urlObject = null;
        try {
            urlObject = new URL(mUrl);
            mUrlConnection = (HttpURLConnection) urlObject.openConnection();
        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        //try { mUrlConnection.setDoOutput(true); } catch (Exception e) {}
        //try { mUrlConnection.setDoInput(true); } catch (Exception e) {}

        for (String key : mRequestProperties.keySet()) {
            mUrlConnection.addRequestProperty(key, mRequestProperties.get(key));
        }

        if (mIgnoreHostnameVerification) {
            trustAllHosts();
            ((HttpsURLConnection) mUrlConnection).setHostnameVerifier(DO_NOT_VERIFY);
        }
    }

    private void setUrlEncoded() {
        mUrlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded;charset=UTF-8");
    }

    private void setRequestMethod(String method) {
        try {
            mUrlConnection.setRequestMethod(method);
        } catch (ProtocolException e1) {
            e1.printStackTrace();
        }
    }

    private void writeBody(String jsonData) {
        if (mAuthToken != null) {
            mUrl = mUrl + "?authentication_token=" + mAuthToken;
        }

        mUrlConnection.setFixedLengthStreamingMode(jsonData.getBytes().length);

        OutputStream outputStream = null;
        try {
            outputStream = mUrlConnection.getOutputStream();
            outputStream.write(jsonData.getBytes());
            outputStream.close();
        } catch (IOException e1) {
            e1.printStackTrace();
        }
    }

    private Response getResponse() {
        Response response = new Response();
        Log.d(LOG_TAG, mUrlConnection.getRequestMethod() + " " + mUrl);

        try {
            response.setStatusCode(mUrlConnection.getResponseCode());
        } catch (IOException e) {
            if (e.getMessage() != null && e.getMessage().contains("authentication challenge")) {
                try {
                    response.setStatusCode(mUrlConnection.getResponseCode());
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
            }
        }

        response.setResponseHeaders(mUrlConnection.getHeaderFields());

        try {
            response.setResponseBody(readStream(mUrlConnection.getInputStream()));
        } catch (Exception e) {
            response.setResponseBody(readStream(mUrlConnection.getErrorStream()));
        } finally {
            if (mUrlConnection != null) {
                mUrlConnection.disconnect();
            }
        }

        return response;
    }

    public Response get() {
        if (mAuthToken != null) {
            mUrl = mUrl + "?authentication_token=" + mAuthToken;
        }
        prepareRequest();

        setRequestMethod("GET");
        setUrlEncoded();

        return getResponse();
    }

    public Response get(HashMap<String, Object> params) {
        String urlParams = UrlParams.getUrlParams(params);
        mUrl = mUrl + "?" + urlParams;

        if (BuildConfig.DEBUG) {
            Log.d(LOG_TAG, "get(HashMap) - URL: " + mUrl);
        }

        prepareRequest();

        setRequestMethod("GET");
        setUrlEncoded();

        return getResponse();
    }

    public long getAsync(HashMap<String, Object> params, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.GET);
        request.setUrlParams(params);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long getAsync(HashMap<String, Object> params) {
        return getAsync(params, 0);
    }

    public long getAsync(int retryCount) {
        return getAsync(null, retryCount);
    }

    public long getAsync() {
        return getAsync(null, 0);
    }

    public Response post(HashMap<String, Object> params) {
        String urlParams = UrlParams.getUrlParams(params);
        mUrl = mUrl + "?" + urlParams;
        prepareRequest();

        setRequestMethod("POST");
        setUrlEncoded();

        return getResponse();
    }

    public Response post(String paramsJson) {
        prepareRequest();

        mUrlConnection.setDoOutput(true);
        mUrlConnection.setDoInput(true);

        setRequestMethod("POST");
        mUrlConnection.setRequestProperty("Content-Type", "application/json");

        writeBody(paramsJson);

        return getResponse();
    }

    public Response postFormData(HashMap<String, Object> params) {
        String formBody = UrlParams.getUrlParams(params);
        prepareRequest();

        mUrlConnection.setDoOutput(true);
        mUrlConnection.setDoInput(true);

        setRequestMethod("POST");
        setUrlEncoded();

        writeBody(formBody);

        return getResponse();
    }

    public long postAsync(HashMap<String, Object> params, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.POST);
        request.setUrlParams(params);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long postAsync(HashMap<String, Object> params) {
        return postAsync(params, 0);
    }

    public long postAsync(String paramsJson, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.POST);
        request.setParamJson(paramsJson);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long postAsync(String paramsJson) {
        return postAsync(paramsJson, 0);
    }

    public Response put(HashMap<String, Object> params) {
        String urlParams = UrlParams.getUrlParams(params);
        mUrl = mUrl + urlParams;
        prepareRequest();

        setRequestMethod("PUT");
        setUrlEncoded();

        return getResponse();
    }

    public Response put(String paramsJson) {
        prepareRequest();

        setRequestMethod("PUT");
        mUrlConnection.setRequestProperty("Content-Type", "application/json");

        writeBody(paramsJson);

        return getResponse();
    }

    public long putAsync(HashMap<String, Object> params, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.PUT);
        request.setUrlParams(params);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long putAsync(HashMap<String, Object> params) {
        return putAsync(params, 0);
    }

    public long putAsync(String paramsJson, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.PUT);
        request.setParamJson(paramsJson);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long putAsync(String paramsJson) {
        return postAsync(paramsJson, 0);
    }

    public Response delete() {
        prepareRequest();

        setRequestMethod("DELETE");
        //setUrlEncoded();

        return getResponse();
    }

    public Response delete(HashMap<String, Object> params) {
        String urlParams = UrlParams.getUrlParams(params);
        mUrl = mUrl + "?" + urlParams;
        prepareRequest();

        setRequestMethod("DELETE");
        //setUrlEncoded();

        return getResponse();
    }

    private void writeParts(ArrayList<Part> parts) {
        mUrlConnection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
        mUrlConnection.setRequestProperty("Connection", "close");

        DataOutputStream request = null;
        try {
            request = new DataOutputStream(mUrlConnection.getOutputStream());

            for (Part part : parts) {
                if (part != null && part.getName() != null && !part.getName().isEmpty()) {
                    if (part.getData() != null && !part.getData().isEmpty()) {
                        request.writeBytes("--" + boundary + "\r\n");
                        request.writeBytes("Content-Disposition: form-data; name=\"" + part.getName() + "\"" + "\r\n");
                        request.writeBytes("\r\n");
                        request.writeBytes(part.getData());
                        request.writeBytes("\r\n");

                    } else if (part.getFile() != null && part.getContentType() != null) {
                        RandomAccessFile file = new RandomAccessFile(part.getFile(), "r");
                        byte[] data = new byte[(int) file.length()];
                        file.read(data);
                        request.writeBytes("--" + boundary + "\r\n");
                        request.writeBytes("Content-Disposition: form-data; name=\"" + part.getName() + "\"; filename=\"" + part.getFile().getName() + "\"" + "\r\n");
                        request.writeBytes("Content-Type: " + part.getContentType() + "\r\n");
                        request.writeBytes("\r\n");
                        request.write(data);
                        request.writeBytes("\r\n");
                    }
                }
            }

            request.writeBytes("--" + boundary + "--" + "\r\n");
            request.close();
        } catch (IOException e) {
            Log.e(LOG_TAG, "Error while creating multi part request body. " + e.getMessage());
        }
    }

    public Response postMultiPart(ArrayList<Part> parts) {
        prepareRequest();
        setRequestMethod("POST");

        writeParts(parts);

        return getResponse();
    }

    public Response putMultiPart(ArrayList<Part> parts) {
        prepareRequest();
        setRequestMethod("PUT");

        writeParts(parts);

        return getResponse();
    }

    public long postMultipartAsync(ArrayList<Part> parts, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.POST);
        request.setMultipart(true);
        request.setParts(parts);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long postMultipartAsync(ArrayList<Part> parts) {
        return postMultipartAsync(parts, 0);
    }

    public long putMultipartAsync(ArrayList<Part> parts, int retryCount) {
        Request request = new Request();
        request.setUrl(mUrl);
        request.setMethod(Method.PUT);
        request.setMultipart(true);
        request.setParts(parts);
        request.setPendingRetryCount(retryCount);

        return RequestQueue.getInstance(mContext).enqueue(request);
    }

    public long putMultipartAsync(ArrayList<Part> parts) {
        return putMultipartAsync(parts, 0);
    }

    /**
     * read response text from http response stream
     *
     * @param inputStream http call response stream
     * @return http response string
     */
    private String readStream(InputStream inputStream) {
        try {
            ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            int c = inputStream.read();
            while (c != -1) {
                byteArrayOutputStream.write(c);
                c = inputStream.read();
            }
            return byteArrayOutputStream.toString();
        } catch (Exception e) {
            return "";
        }
    }

    /**
     * The below code is used to trust the https connection
     * without verifying the hostname. This is used for testing
     * purpose only, in development mode.
     */

    final static HostnameVerifier DO_NOT_VERIFY = new HostnameVerifier() {
        public boolean verify(String hostname, SSLSession session) {
            return true;
        }
    };

    /**
     * Trust every server - dont check for any certificate
     */
    private static void trustAllHosts() {

        try {
            HttpsURLConnection.setDefaultHostnameVerifier(new HostnameVerifier() {
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            });
            SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, new X509TrustManager[]{new X509TrustManager() {
                public void checkClientTrusted(X509Certificate[] chain,
                                               String authType) throws CertificateException {
                }

                public void checkServerTrusted(X509Certificate[] chain,
                                               String authType) throws CertificateException {
                }

                public X509Certificate[] getAcceptedIssuers() {
                    return new X509Certificate[0];
                }
            }}, new SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(
                    context.getSocketFactory());
        } catch (Exception e) { // should never happen
            e.printStackTrace();
        }

        // Create a trust manager that does not validate certificate chains
        /* old implementation
        TrustManager[] trustAllCerts = new TrustManager[] { new X509TrustManager() {
            public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                return new java.security.cert.X509Certificate[] {};
            }

            public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {}

            public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {}
        } };

        // Install the all-trusting trust manager
        try {
            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            HttpsURLConnection
                    .setDefaultSSLSocketFactory(sc.getSocketFactory());
        } catch (Exception e) {
            e.printStackTrace();
        }*/
    }
}
