package ch.awae.utils.statemachine;

import java.util.ArrayList;
import java.util.Objects;
import java.util.UUID;
import java.util.logging.Logger;

/**
 * Builder for constructing state machines.
 * <p>
 * A state machine consists of one or more state machine cores. There cores are
 * constructed using a {@link MachineCoreBuilder}. State machines support 2
 * modes for handling inner events: <em>normal mode</em> and <em>priority
 * mode</em>. In normal mode events generated by transitions of the machine
 * cores are simply added to the main event queue <em>behind</em> already
 * enqueued requests. In priority mode these <em>internal events</em> are
 * handled before any already enqueued event. From the internal events the
 * oldest ones are always processed first. A new <em>external event</em> is only
 * processed once all internal events have been handled and no new internal
 * events have been created. The mode can be set using
 * {@link #setPrioritiseInternalEvents(boolean)}.
 * </p>
 * 
 * @author Andreas Wälchli
 * @since awaeUtils 0.0.3
 * @version 1.3 (0.0.5)
 * 
 * @see StateMachine
 * @see MachineCoreBuilder
 */
public class StateMachineBuilder {

    private ArrayList<MachineCoreBuilder> cores                    = new ArrayList<>();
    private boolean                       prioritiseInternalEvents = false;

    /**
     * creates a new empty builder
     */
    public StateMachineBuilder() {
        super();
    }

    /**
     * copy constructor
     * 
     * @param builder
     *            the builder to copy
     * @throws NullPointerException
     *             {@code builder} is {@code null}
     * @since 1.2
     */
    public StateMachineBuilder(StateMachineBuilder builder) {
        Objects.requireNonNull(builder);
        prioritiseInternalEvents = builder.prioritiseInternalEvents;
        synchronized (builder.cores) {
            cores.addAll(builder.cores);
        }
    }

    /**
     * Adds a {@link MachineCoreBuilder} to this builder. That builder will be
     * used to derive a state machine core internally. The passed
     * {@link MachineCoreBuilder} instance will be copied and will therefore not
     * be affected by further changes to that instance.
     * 
     * @param builder
     *            the core builder to add. may not be {@code null}
     * @return the builder itself
     * @throws NullPointerException
     *             {@code builder} is {@code null}
     */
    public StateMachineBuilder addMachineCore(MachineCoreBuilder builder) {
        Objects.requireNonNull(builder);
        MachineCoreBuilder bldr = builder.copy();
        synchronized (cores) {
            cores.add(bldr);
        }
        return this;
    }

    /**
     * creates a copy of this builder
     * 
     * @return the copy
     * @since 1.2
     */
    public StateMachineBuilder copy() {
        return new StateMachineBuilder(this);
    }

    /**
     * Set if internal events should get priority over all other events. if set
     * to {@code true} internal events generated by an event are processed
     * immediately before processing the next external event. If the processing
     * of internal events yields more internal events they are queued in order
     * of appearance (i.e. the oldest internal event is always processed first)
     * 
     * @param prioritise
     * @return the builder itself
     * 
     * @since 1.3
     */
    public StateMachineBuilder setPrioritiseInternalEvents(boolean prioritise) {
        prioritiseInternalEvents = prioritise;
        return this;
    }

    /**
     * Constructs a {@link StateMachine} represented by this builder and all its
     * {@link MachineCoreBuilder MachineCoreBuilders}
     * 
     * @return a state machine
     * @throws IllegalArgumentException
     *             a core could not be constructed due to invalid data
     */
    public StateMachine build() {
        String uuid = UUID.randomUUID().toString();
        Logger logger = Logger.getLogger("ch.awae.utils.statemachine.StateMachine");
        MachineCore[] cores = new MachineCore[this.cores.size()];
        logger.fine("initialising new state machine " + uuid);
        logger.finer(uuid + ": loading " + cores.length + " cores");
        for (int i = 0; i < cores.length; i++) {
            try {
                logger.finer(uuid + ": loading core " + i + " (" + (i + 1) + "/" + cores.length + ")");
                cores[i] = this.cores.get(i).build(i, uuid + "[" + i + "]", logger);
            } catch (
                    NullPointerException
                    | IllegalArgumentException ex) {
                logger.severe(uuid + ": core " + i + " could not be constructed due to error: " + ex.getMessage());
                throw new IllegalArgumentException("core[" + i + "] could not be constructed:\n" + ex.getMessage(), ex);
            }
            logger.finer(uuid + ": loaded core " + i);
        }
        logger.finer("loaded machine " + uuid + " with " + cores.length + " cores");
        return new StateMachineImpl(uuid, prioritiseInternalEvents, logger, cores);
    }

}
