/*
 *   Copyright 2011, 2012 Hauser Olsson GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * Package: ch.agent.t2.time
 * Type: TimeIndex
 * Version: 1.0.1
 */
package ch.agent.t2.time;

import ch.agent.t2.T2Exception;

/**
 * A TimeIndex object represents a time, or more precisely, a point in time. 
 * It consists of two elements:
 * <ol>
 * <li>a time domain,
 * <li>a numerical index.
 * </ol>
 * When required, the index is interpreted or generated by the time domain. However there are
 * important and frequent operations which do not require interpretation and which can
 * be performed efficiently using integer addition. An example is the
 * creation of a new time index object by incrementing another one.
 * <p>
 * The design assumes that TimeIndex is implemented as an immutable object.
 * 
 * @author Jean-Paul Vetterli
 * @version 1.0.1
 * @see TimeDomain
 */
public interface TimeIndex extends Comparable<TimeIndex> {

	/**
	 * Return the time domain.
	 * 
	 * @return the time domain
	 */
	TimeDomain getTimeDomain();

	/**
	 * Return the numeric index.
	 * 
	 * @return the numeric index
	 */
	long asLong();
	
	/**
	 * Return the time as an offset from the origin. An exception is thrown if
	 * the offset does not fit into a 32 bit integer.
	 * 
	 * @return the time as an offset from the origin
	 * @throws T2Exception
	 */
	int asOffset() throws T2Exception;
	
	/**
	 * Return the day of the week. An exception is thrown if the time domain 
	 * resolution is YEAR or MONTH
	 * 
	 * @return the day of the week
	 * @throws T2Exception
	 */
	DayOfWeek getDayOfWeek() throws T2Exception;

	/**
	 * Return the TimeIndex for a day defined by name and rank within a reference period.
	 * A reference period is the current month or the current year.
	 * If the rank is negative, days are counted from the end, with -1
	 * meaning the last day. The result is a TimeIndex with DAY resolution.
	 * 
	 * @param referencePeriod one of the values MONTH or YEAR
	 * @param day the day name
	 * @param rank the day rank
	 * @return a time index with DAY resolution
	 * @throws T2Exception
	 */
	TimeIndex getDayByRank(Resolution referencePeriod, DayOfWeek day, int rank) throws T2Exception;
	
	/**
	 * Return the microsecond component of the time.
	 * 
	 * @return a number between 0 and 999999
	 */
	int getMicrosecond();
	
	/**
	 * Return the second component of the time.
	 * 
	 * @return a number between 0 and 59
	 */
	int getSecond();
	
	/**
	 * Return the minute component of the time.
	 * 
	 * @return a number between 0 and 59
	 */
	int getMinute();
	
	/**
	 * Return the hour component of the time.
	 * 
	 * @return a number between 0 and 23.
	 */
	int getHour();

	/**
	 * Return the day component of the time.
	 * 
	 * @return a number between 1 and the number of days in the current month and year
	 */
	int getDay();

	/**
	 * Return the month component of the time.
	 * 
	 * @return a number between 1 and 12
	 */
	int getMonth();
	
	/**
	 * Return the year component of the time.
	 * 
	 * @return a number between 0 and the largest long
	 */
	long getYear();
	
	/**
	 * Return a new time index with the time incremented from this one. For
	 * respective increments 1 and -1, the time returned correspond to the next
	 * and previous periods.
	 * An exception is thrown when the increment cannot be applied without
	 * causing an overflow.
	 * <p>
	 * This method is related to {@link #sub(TimeIndex) sub(TimeIndex)}.
	 * 
	 * @param increment positive or negative number of units to add
	 * @return a new TimeIndex object
	 * @throws T2Exception
	 */
	TimeIndex add(long increment) throws T2Exception;

	/**
	 * Return the number of time units since the time specified as argument.
	 * The result is negative if the argument corresponds to a later time.
	 * An exception is thrown if the domains are not equal.
	 * <p>
	 * This method is related to {@link #add(long) add(long)} in the following way:
	 * <xmp>t1.add(t2.sub(t1)).equals(t2) 
	 * </xmp>
	 * 
	 * @param time a non-null time index to substract from this time index
	 * @return the number of periods between this time index and the argument, positive is the argument is earlier
	 * @throws T2Exception
	 */
	long sub(TimeIndex time) throws T2Exception;

	/**
	 * Return a new time index by converting this one to the given time domain. No
	 * adjustment is allowed.
	 * 
	 * @param domain a non-null time domain
	 * @return a time index in the given domain
	 * @throws T2Exception
	 */
	TimeIndex convert(TimeDomain domain) throws T2Exception;

	/**
	 * Return a new time index by converting this one to the given time domain, 
	 * possibly adjusting the time as specified.
	 * 
	 * @param domain a non-null time domain
	 * @param adjustment a non-null allowed adjustment 
	 * @return a time index in the given domain
	 * @throws T2Exception
	 */
	TimeIndex convert(TimeDomain domain, Adjustment adjustment) throws T2Exception;
	
	/**
	 * Return a string representation of the time. The order of the time
	 * components is: year, month, day, hour, minute, second, microsecond.
	 * Because the format string is a Formatter format, it is possible to
	 * reorder these components.
	 * <p>
	 * For example <code>"%04d%02d%02d"</code> would produce 19700401 and
	 * <code>"%3$d.%2$d.%1$04d"</code> would produce 1.4.1970.
	 * <p>
	 * As a special trick, when the format string is empty, the date is returned
	 * in the compact format d.m.yy, with the day and month without a leading zero, and
	 * the year starting at the 3d digit.
	 * <p>
	 * If the argument is null, formatting is delegated to {@link ExternalTimeFormat}.
	 * <p>
	 * In case of errors in the format string, an IllegalFormatException is thrown.
	 *  
	 * @param format
	 *            a format string, or an empty string, or null
	 * @return a string representation of the time
	 */
	String toString(String format);
	
}