/*
 * Decompiled with CFR 0.152.
 */
package ch.aaap.harvestclient.impl;

import ch.aaap.harvestclient.exception.ForbiddenException;
import ch.aaap.harvestclient.exception.HarvestHttpException;
import ch.aaap.harvestclient.exception.HarvestRuntimeException;
import ch.aaap.harvestclient.exception.InvalidAuthorizationException;
import ch.aaap.harvestclient.exception.NotFoundException;
import ch.aaap.harvestclient.exception.RateLimitedException;
import ch.aaap.harvestclient.exception.RequestProcessingException;
import ch.aaap.harvestclient.exception.ServerErrorException;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.concurrent.TimeUnit;
import okhttp3.Headers;
import okhttp3.ResponseBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit2.Call;
import retrofit2.Response;

public class ExceptionHandler {
    private static final Logger log = LoggerFactory.getLogger(ExceptionHandler.class);
    private static final int MAX_RETRY_COUNT = 3;
    private static final int MAX_WAIT = 30;

    public static <T> T callOrThrow(Call<T> call) {
        return ExceptionHandler.callOrThrow(call, 1);
    }

    public static <T> T callOrThrow(Call<T> call, int attemptCount) {
        try {
            log.debug("Executing call {}", (Object)call.request());
            Response response = call.execute();
            int code = response.code();
            if (response.isSuccessful()) {
                log.debug("Success [{}]", (Object)code);
                return (T)response.body();
            }
            log.warn("Failure -> {}", (Object)code);
            ResponseBody errorBody = response.errorBody();
            switch (code) {
                case 401: {
                    throw new InvalidAuthorizationException(errorBody);
                }
                case 403: {
                    throw new ForbiddenException(errorBody);
                }
                case 404: {
                    throw new NotFoundException(errorBody);
                }
                case 422: {
                    throw new RequestProcessingException(errorBody);
                }
                case 429: {
                    return ExceptionHandler.retryLater(call, response, attemptCount);
                }
                case 500: {
                    throw new ServerErrorException(errorBody, 500);
                }
            }
            throw new HarvestHttpException(errorBody, code);
        }
        catch (IOException e) {
            throw new HarvestRuntimeException(e);
        }
    }

    private static <T> T retryLater(Call<T> failedCall, Response<T> failedResponse, int attemptCount) {
        if (attemptCount > 3) {
            throw new RateLimitedException(failedResponse.errorBody());
        }
        ++attemptCount;
        int wait = 0;
        if (failedResponse.code() == 429) {
            int secondsToWait = ExceptionHandler.parseRetryAfter(failedResponse);
            wait = Math.min(secondsToWait, 30);
        }
        try {
            log.warn("Waiting {} seconds to retry call {}", (Object)wait, (Object)failedCall.request());
            Thread.sleep(TimeUnit.SECONDS.toMillis(wait));
        }
        catch (InterruptedException e) {
            throw new HarvestRuntimeException(e);
        }
        Call call = failedCall.clone();
        return ExceptionHandler.callOrThrow(call, attemptCount);
    }

    private static <T> int parseRetryAfter(Response<T> failedResponse) {
        int wait = 1;
        Headers headers = failedResponse.headers();
        String header = headers.get("Retry-After");
        if (header == null) {
            log.error("Missing Retry-After Header for {}", failedResponse);
            return wait;
        }
        try {
            return Integer.parseInt(header);
        }
        catch (NumberFormatException e) {
            try {
                ZonedDateTime date = ZonedDateTime.parse(header, DateTimeFormatter.RFC_1123_DATE_TIME);
                log.debug("Retry-After is {} ", (Object)date);
                return (int)Duration.between(Instant.now(), date).getSeconds();
            }
            catch (DateTimeParseException e1) {
                log.error("Invalid Retry-After Header for {}", failedResponse);
                return wait;
            }
        }
    }
}

