package cc.concurrent.config.server.service;

import cc.concurrent.config.core.model.FilePublished;
import cc.concurrent.config.core.util.Md5Util;
import cc.concurrent.config.server.dao.AppDao;
import cc.concurrent.config.server.dao.FileCurrentDao;
import cc.concurrent.config.server.dao.FileHistoryDao;
import cc.concurrent.config.server.dao.FilePublishedDao;
import cc.concurrent.config.server.model.App;
import cc.concurrent.config.server.model.FileCurrent;
import cc.concurrent.config.server.model.FileHistory;
import cc.concurrent.config.server.util.*;
import com.google.common.base.Charsets;
import com.google.common.hash.Hashing;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.DigestUtils;

import java.util.Date;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * User: yanghe.liang
 * Date: 13-10-13
 * Time: 下午4:44
 */
@Component
public class FileService {

    @Autowired
    FileCurrentDao fileCurrentDao;

    @Autowired
    FileHistoryDao fileHistoryDao;

    @Autowired
    AppDao appDao;

    @Autowired
    FilePublishedDao filePublishedDao;

    public void addFile(String appName, String fileName, String xml, String comment) {
        checkAppName(appName);
        ValidUtil.checkRegex("[_0-9a-zA-Z]{3,20}", fileName, "fileName需要匹配正则表达式[_0-9a-zA-Z]{3,20}");
        ValidUtil.checkRegex(".{1,100}", comment, "注释的长度必须在1到100内");
        String formatXml = XmlUtil.format(xml);

        int version = 1;
        String username = FlashRequestContext.getCurrentContext().getUsername();
        FileCurrent fileCurrent = new FileCurrent(appName, fileName, version, formatXml, comment, username, new Date(), 0, null, null);
        FileHistory fileHistory = new FileHistory(appName, fileName, version, formatXml, comment, username, new Date(), false, null, null);
        fileCurrentDao.addFileCurrent(fileCurrent);
        fileHistoryDao.addFileHistory(fileHistory);
    }

    public List<FileCurrent> getFileCurrents(String appName) {
        checkAppName(appName);
        return fileCurrentDao.getFileCurrents(appName);
    }

    public void updateFile(String appName, String fileName, int version, String xml, String comment) {
        checkAppName(appName);
        ValidUtil.checkRegex("[_0-9a-zA-Z]{3,20}", fileName, "fileName需要匹配正则表达式[_0-9a-zA-Z]{3,20}");
        ValidUtil.checkRegex(".{1,100}", comment, "注释的长度必须在1到100内");
        String formatXml = XmlUtil.format(xml);

        FileCurrent oldFileCurrent = fileCurrentDao.getFileCurrent(appName, fileName);
        checkArgument(oldFileCurrent.getVersion() == version, "only latest version can update");

        String username = FlashRequestContext.getCurrentContext().getUsername();
        FileCurrent fileCurrent = new FileCurrent(appName, fileName, version + 1, formatXml, comment, username, new Date(),
                oldFileCurrent.getLastPublishVersion(), oldFileCurrent.getLastPublisher(), oldFileCurrent.getLastPublishTime());
        FileHistory fileHistory = new FileHistory(appName, fileName, version + 1, formatXml, comment, username, new Date(), false, null, null);
        fileCurrentDao.updateFileCurrent(fileCurrent);
        fileHistoryDao.addFileHistory(fileHistory);
    }

    public void publishFile(String appName, String fileName, int version) {
        checkAppName(appName);
        FileCurrent fileCurrent = fileCurrentDao.getFileCurrent(appName, fileName);
        checkNotNull(fileCurrent);
        checkArgument(fileCurrent.getVersion() == version, "only latest version can update");
        FileHistory fileHistory = fileHistoryDao.getFileHistory(appName, fileName, version);
        checkNotNull(fileHistory);
        checkArgument(!fileHistory.isPublished(), "has published");

        String xml = fileHistory.getXml();
        String md5 = Md5Util.md5(xml);
        FilePublished filePublished = new FilePublished(appName, fileName, version, xml, md5);
        if (filePublishedDao.updateFilePublished(filePublished) == 0) {
            filePublishedDao.addFilePublished(filePublished);
        }

        String username = FlashRequestContext.getCurrentContext().getUsername();
        fileCurrent.setLastPublisher(username);
        fileCurrent.setLastPublishTime(new Date());
        fileCurrent.setLastPublishVersion(version);
        fileCurrentDao.updateFileCurrent(fileCurrent);
        fileHistoryDao.updatePublish(appName, fileName, version, true, username, new Date());
    }


    public List<FileHistory> getFileHistories(String appName, String fileName) {
        checkAppName(appName);
        return fileHistoryDao.getFileHistories(appName, fileName);
    }

    public FileHistory getFileHistory(String appName, String fileName, int version) {
        checkAppName(appName);
        FileHistory fileHistory = fileHistoryDao.getFileHistory(appName, fileName, version);
        checkNotNull(fileHistory, "app[%s] file[%s] version[%s] don't exsit", appName, fileName, version);

        FileCurrent fileCurrent = fileCurrentDao.getFileCurrent(appName, fileName);
        boolean isLatest = fileCurrent.getVersion() == version ? true : false;
        fileHistory.setLatest(isLatest);

        return fileHistory;
    }

    public HtmlView diff(String appName, String fileName, int newVersion, int oldVersion) {
        checkAppName(appName);
        List<FileHistory> fileHistories = fileHistoryDao.getFileHistories(appName, fileName);
        String newXml = null;
        String oldXml = null;
        for (FileHistory fileHistory : fileHistories) {
            if (fileHistory.getVersion() == newVersion) {
                newXml = fileHistory.getXml();
            } else if (fileHistory.getVersion() == oldVersion) {
                oldXml = fileHistory.getXml();
            }
        }
        checkNotNull(newXml, "newVersion %s don't exist", newVersion);
        checkNotNull(oldXml, "oldVersion %s don't exist", oldVersion);
        HtmlView htmlView = DiffUtil.diff(newXml, oldXml);
        return htmlView;
    }

    private void checkAppName(String appName) {
        App app = appDao.getApp(appName);
        checkArgument(app != null, "app %s don't exsit", appName);
    }

}





