package cc.concurrent.config.server.controller;

import cc.concurrent.config.server.model.FileCurrent;
import cc.concurrent.config.server.model.FileHistory;
import cc.concurrent.config.server.service.FileService;
import cc.concurrent.config.server.util.HtmlView;
import org.dom4j.DocumentException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;

import java.io.IOException;
import java.util.List;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * User: yanghe.liang
 * Date: 13-10-13
 * Time: 下午4:25
 */
@Controller
public class FileController {

    @Autowired
    FileService fileService;

    /**
     * 显示某个app的文件列表
     */
    @RequestMapping(value = "/{appName}", method = RequestMethod.GET)
    public String files(@PathVariable("appName") String appName, Model model) {
        List<FileCurrent> fileCurrents = fileService.getFileCurrents(appName);
        model.addAttribute("appName", appName);
        model.addAttribute("fileCurrents", fileCurrents);
        return "files";
    }

    /**
     * 文件的所有历史记录
     */
    @RequestMapping(value = "/{appName}/{fileName}", method = RequestMethod.GET)
    public String fileHistories(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName, Model model) {
        List<FileHistory> fileHistories = fileService.getFileHistories(appName, fileName);
        model.addAttribute("appName", appName);
        model.addAttribute("fileName", fileName);
        model.addAttribute("fileHistories", fileHistories);
        return "fileHistories";
    }

    /**
     * 文件添加页
     */
    @RequestMapping(value = "/{appName}/addFile", method = RequestMethod.GET)
    public String addFile(@PathVariable("appName") String appName, Model model) {
        model.addAttribute("appName", appName);
        return "addFile";
    }

    /**
     * 执行文件添加
     */
    @RequestMapping(value = "/{appName}/addFile", method = RequestMethod.POST)
    public String addFile(@PathVariable("appName") String appName, String fileName, String xml, String comment, Model model)
            throws IOException, DocumentException {

        try {
            fileService.addFile(appName, fileName, xml, comment);
            return "redirect:/" + appName;
        } catch (IllegalArgumentException e) {
            model.addAttribute("appName", appName);
            model.addAttribute("fileName", fileName);
            model.addAttribute("xml", xml);
            model.addAttribute("comment", comment);
            model.addAttribute("errorMsg", e.getMessage());
            return "addFile";
        }
    }

    /**
     * 某个版本的文件
     */
    @RequestMapping(value = "/{appName}/{fileName}/{version}", method = RequestMethod.GET)
    public String file(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName,
                       @PathVariable("version") int version, Model model) {
        FileHistory fileHistory = fileService.getFileHistory(appName, fileName, version);
        model.addAttribute("appName", appName);
        model.addAttribute("fileName", fileName);
        model.addAttribute("version", version);
        model.addAttribute("fileHistory", fileHistory);
        return "file";
    }

    /**
     * 文件更新页
     */
    @RequestMapping(value = "/{appName}/{fileName}/{version}/update", method = RequestMethod.GET)
    public String updateFile(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName,
                             @PathVariable("version") int version, Model model) {
        FileHistory fileHistory  = fileService.getFileHistory(appName, fileName, version);
        checkArgument(fileHistory.isLatest(), "only latest version can update");
        model.addAttribute("appName", appName);
        model.addAttribute("fileName", fileName);
        model.addAttribute("version", fileHistory.getVersion());
        model.addAttribute("xml", fileHistory.getXml());
        return "updateFile";
    }

    /**
     * 执行文件更新
     */
    @RequestMapping(value = "/{appName}/{fileName}/{version}/update", method = RequestMethod.POST)
    public String updateFile(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName,
                             @PathVariable("version") int version,  String xml, String comment, Model model) {
        try {
            fileService.updateFile(appName, fileName, version, xml, comment);
            return "redirect:/" + appName;
        } catch (IllegalArgumentException e) {
            model.addAttribute("appName", appName);
            model.addAttribute("fileName", fileName);
            model.addAttribute("version", version);
            model.addAttribute("xml", xml);
            model.addAttribute("comment", comment);
            model.addAttribute("errorMsg", e.getMessage());
            return "updateFile";
        }
    }

    /**
     * 发布文件
     */
    @RequestMapping(value = "/{appName}/{fileName}/{version}/publish", method = RequestMethod.GET)
    public String updateFile(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName,
                             @PathVariable("version") int version) {
        fileService.publishFile(appName, fileName, version);
        return "redirect:/" + appName;
    }

    /**
     * 对比文件的历史版本
     */
    @RequestMapping(value = "/{appName}/{fileName}/{newVersion}/diff/{oldVersion}", method = RequestMethod.GET)
    public String diff(@PathVariable("appName") String appName, @PathVariable("fileName") String fileName,
                             @PathVariable("newVersion") int newVersion,  @PathVariable("oldVersion") int oldVersion,
                             Model model) {
        checkArgument(newVersion > oldVersion, "newVersion must larger than oldVersion");
        HtmlView htmlView = fileService.diff(appName, fileName, newVersion, oldVersion);
        model.addAttribute("appName", appName);
        model.addAttribute("fileName", fileName);
        model.addAttribute("newVersion", newVersion);
        model.addAttribute("oldVersion", oldVersion);
        model.addAttribute("newHtml", htmlView.getNewHtml());
        model.addAttribute("oldHtml", htmlView.getOldHtml());
        return "diff";
    }

}
