/*
 * Copyright 2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package capital.scalable.restdocs.payload;

import java.lang.reflect.GenericArrayType;
import java.lang.reflect.Type;
import java.util.Map;

import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.web.method.HandlerMethod;

public class JacksonResponseFieldSnippet extends AbstractJacksonFieldSnippet {

    public static final String RESPONSE_FIELDS = "auto-response-fields";
    public static final String SPRING_DATA_PAGE_CLASS = "org.springframework.data.domain.Page";

    private final Type responseBodyType;
    private final boolean failOnUndocumentedFields;

    public JacksonResponseFieldSnippet() {
        this(null, false);
    }

    public JacksonResponseFieldSnippet(Type responseBodyType, boolean failOnUndocumentedFields) {
        super(RESPONSE_FIELDS, null);
        this.responseBodyType = responseBodyType;
        this.failOnUndocumentedFields = failOnUndocumentedFields;
    }

    public JacksonResponseFieldSnippet responseBodyAsType(Type responseBodyType) {
        return new JacksonResponseFieldSnippet(responseBodyType, failOnUndocumentedFields);
    }

    public JacksonResponseFieldSnippet failOnUndocumentedFields(boolean failOnUndocumentedFields) {
        return new JacksonResponseFieldSnippet(responseBodyType, failOnUndocumentedFields);
    }

    @Override
    protected Type getType(final HandlerMethod method) {
        if (responseBodyType != null) {
            return responseBodyType;
        }

        Class<?> returnType = method.getReturnType().getParameterType();
        if (returnType == ResponseEntity.class) {
            return firstGenericType(method.getReturnType());
        } else if (returnType == HttpEntity.class) {
            return firstGenericType(method.getReturnType());
        } else if (SPRING_DATA_PAGE_CLASS.equals(returnType.getCanonicalName())) {
            return firstGenericType(method.getReturnType());
        } else if (isCollection(returnType)) {
            return new GenericArrayType() {

                @Override
                public Type getGenericComponentType() {
                    return firstGenericType(method.getReturnType());
                }
            };
        } else if ("void".equals(returnType.getName())) {
            return null;
        } else {
            return returnType;
        }
    }

    @Override
    protected void enrichModel(Map<String, Object> model, HandlerMethod handlerMethod) {
        model.put("isPageResponse", isPageResponse(handlerMethod));
    }

    private boolean isPageResponse(HandlerMethod handlerMethod) {
        return SPRING_DATA_PAGE_CLASS.equals(
                handlerMethod.getReturnType().getParameterType().getCanonicalName());
    }

    @Override
    public String getHeader() {
        return "Response fields";
    }

    @Override
    protected boolean shouldFailOnUndocumentedFields() {
        return failOnUndocumentedFields;
    }
}
