package ca.genovese.coffeecats.cats.syntax;

import ca.genovese.coffeecats.cats.Apply;
import ca.genovese.coffeecats.util.Kind;
import ca.genovese.coffeecats.util.Unit;
import ca.genovese.coffeecats.util.types.function.Function3;
import ca.genovese.coffeecats.util.types.tuple.Tuple2;

import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

public interface ApplyOps<F, A> extends FunctorOps<F, A> {
  static <F, A> ApplyOps<F, A> create(Apply<F> f, Kind<F, A> fb) {
    return new ApplyOps<F, A>() {
      @Override
      public Apply<F> F() {
        return f;
      }

      @Override
      public Kind<F, A> get() {
        return fb;
      }
    };
  }

  default <B> ApplyOps<F, B> createLocal(Kind<F, B> fb) {
    return ApplyOps.create(F(), fb);
  }


  @Override
  Apply<F> F();

  /*
   * Given a value and a function in the Apply context, applies the
   * function to the value.
   */
  default <B> ApplyOps<F, B> apply(Kind<F, Function<A, B>> f) {
    return create(F(), F().apply(get(), f));
  }

  /*
   * apply2 is a binary version of apply, defined in terms of apply.
   */
  default <B, Z> ApplyOps<F, Z> apply2(Kind<F, B> fb, Kind<F, BiFunction<A, B, Z>> ff) {
    return createLocal(F().apply2(get(), fb, ff));
  }


  /*
   * Applies the pure (binary) function f to the effectful values fa and fb.
   * <p/>
   * map2 can be seen as a binary version of [[cats.Functor]]#map.
   */
  default <B, Z> ApplyOps<F, Z> map2(Kind<F, A> fa, Kind<F, B> fb, BiFunction<A, B, Z> f) {
    return createLocal(F().map2(get(), fb, f));
  }

  /*
   * Applies the pure (binary) function f to the effectful values fa, fb, and fc.
   */
  default <B, C, Z> ApplyOps<F, Z> map3(Kind<F, B> fb, Kind<F, C> fc, Function3<A, B, C, Z> f) {
    return createLocal(F().map3(get(), fb, fc, f));
  }

  @Override
  default <B> ApplyOps<F, B> map(Function<A, B> f) {
    return createLocal(F().map(get(), f));
  }

  @Override
  default <B> ApplyOps<F, Tuple2<A, B>> zipWith(Function<A, B> f) {
    return createLocal(F().zipWith(get(), f));
  }

  @Override
  default <B> ApplyOps<F, B> as(Supplier<B> b) {
    return createLocal(F().as(get(), b));
  }

  @Override
  default ApplyOps<F, Unit> voidMe() {
    return createLocal(F().voidMe(get()));
  }

  @Override
  default <B> ApplyOps<F, B> imap(Function<A, B> f, Function<B, A> g) {
    return createLocal(F().imap(get(), f, g));
  }
}
