package ca.genovese.coffeecats.algebra;

import java.util.function.Function;

/*
 * A type class used to determine equality between 2 instances of the same
 * type. Any 2 instances `x` and `y` are equal if `eqv(x, y)` is `true`.
 * Moreover, `eqv` should form an equivalence relation.
 */
public interface Eq<A> {

  /*
   * Returns `true` if `x` and `y` are equivalent, `false` otherwise.
   *
   * @param x first item to be compared
   * @param y seconds item to be compared
   * @return result of the comparison
   */
  Boolean eqv(A x, A y);

  /*
   * Returns `false` if `x` and `y` are equivalent, `true` otherwise.

   * @param x first item to be compared
   * @param y seconds item to be compared
   * @return result of the comparison
   */
  default Boolean neqv(A x, A y) {
    return !eqv(x, y);
  }

  /*
   * Constructs a new `Eq` instance for type `B` where 2 elements are
   * equivalent iff `eqv(f(x), f(y))`.
   */
  /*
   * Constructs a new `Eq` instance for type `B` where 2 elements are
   * equivalent iff `eqv(f(x), f(y))`.
   *
   * @param f Function to convert from a B to an A
   * @param <B> Type for the new Eq derived from this Eq
   * @return a new Eq&lt;B&gt;
   */
  default <B> Eq<B> on(Function<B, A> f) {
    return new Eq<B>() {
      @Override
      public Boolean eqv(B x, B y) {
        return Eq.this.eqv(f.apply(x), f.apply(y));
      }
    };
  }
}
