@file:JvmName("Logging")

package ca.deprecatedlogic.logging

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import kotlin.reflect.KClass
import kotlin.reflect.full.findAnnotation

private var INSTANCE: Lazy<Logger>? = null

private val Loggable.factory: () -> Logger
    get() = {
        val annotation = this::class.findAnnotation<LoggerName>()
        if (annotation == null) {
            val name = this::class.simpleName
            if (name == null) {
                LoggerFactory.getLogger(this::class.java)
            }
            else {
                LoggerFactory.getLogger(name)
            }
        }
        else {
            LoggerFactory.getLogger(annotation.name)
        }
    }

private val Loggable.instance: Lazy<Logger>
    get() {
        synchronized(this) {
            if (INSTANCE == null) {
                INSTANCE = lazy(factory)
            }
            return INSTANCE!!
        }
    }

/**
 * Flags an implementor as requiring access to the logging capabilities of a [Logger].
 */
@Suppress("unused")
interface Loggable {
    /**
     * The actual [Logger] instance that all logging functions are delegated to.
     *
     * Attempts to find a reasonable logger name to use by [LoggerName] annotation or [KClass.simpleName].
     */
    val logger: Logger
        get() = instance.value
    
    val isTraceEnabled: Boolean
        get() = logger.isTraceEnabled
    
    val isDebugEnabled: Boolean
        get() = logger.isDebugEnabled
    
    val isInfoEnabled: Boolean
        get() = logger.isInfoEnabled
    
    val isWarnEnabled: Boolean
        get() = logger.isWarnEnabled
    
    val isErrorEnabled: Boolean
        get() = logger.isErrorEnabled
    
    
    fun trace(message: String): Unit {
        if (isTraceEnabled) {
            logger.trace(message)
        }
    }
    
    fun debug(message: String): Unit {
        if (isDebugEnabled) {
            logger.debug(message)
        }
    }
    
    fun info(message: String): Unit {
        if (isInfoEnabled) {
            logger.info(message)
        }
    }
    
    fun warn(message: String): Unit {
        if (isWarnEnabled) {
            logger.warn(message)
        }
    }
    
    fun error(message: String): Unit {
        if (isErrorEnabled) {
            logger.error(message)
        }
    }
    
    
    fun trace(message: String, throwable: Throwable): Unit {
        if (isTraceEnabled) {
            logger.trace(message, throwable)
        }
    }
    
    fun debug(message: String, throwable: Throwable): Unit {
        if (isDebugEnabled) {
            logger.debug(message, throwable)
        }
    }
    
    fun info(message: String, throwable: Throwable): Unit {
        if (isInfoEnabled) {
            logger.info(message, throwable)
        }
    }
    
    fun warn(message: String, throwable: Throwable): Unit {
        if (isWarnEnabled) {
            logger.warn(message, throwable)
        }
    }
    
    fun error(message: String, throwable: Throwable): Unit {
        if (isErrorEnabled) {
            logger.error(message, throwable)
        }
    }
    

    fun trace(message: () -> String): Unit {
        if (isTraceEnabled) {
            logger.trace(message())
        }
    }

    fun debug(message: () -> String): Unit {
        if (isDebugEnabled) {
            logger.debug(message())
        }
    }

    fun info(message: () -> String): Unit {
        if (isInfoEnabled) {
            logger.info(message())
        }
    }

    fun warn(message: () -> String): Unit {
        if (isWarnEnabled) {
            logger.warn(message())
        }
    }

    fun error(message: () -> String): Unit {
        if (isErrorEnabled) {
            logger.error(message())
        }
    }
    

    fun trace(message: () -> String, throwable: Throwable): Unit {
        if (isTraceEnabled) {
            logger.trace(message(), throwable)
        }
    }

    fun debug(message: () -> String, throwable: Throwable): Unit {
        if (isDebugEnabled) {
            logger.debug(message(), throwable)
        }
    }

    fun info(message: () -> String, throwable: Throwable): Unit {
        if (isInfoEnabled) {
            logger.info(message(), throwable)
        }
    }

    fun warn(message: () -> String, throwable: Throwable): Unit {
        if (isWarnEnabled) {
            logger.warn(message(), throwable)
        }
    }

    fun error(message: () -> String, throwable: Throwable): Unit {
        if (isErrorEnabled) {
            logger.error(message(), throwable)
        }
    }
}
