package ca.deprecatedlogic.logging

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import kotlin.reflect.KClass

@Suppress("unused")
interface Loggable {
    private companion object {
        val instance: Lazy<Logger> = lazy {
            val annotation = this::class.annotations.singleOrNull {
                it is LoggerName
            }
            if (annotation == null) {
                val name = this::class.simpleName
                if (name == null) {
                    LoggerFactory.getLogger(this::class.java)
                }
                else {
                    LoggerFactory.getLogger(name)
                }
            }
            else {
                annotation as LoggerName
                LoggerFactory.getLogger(annotation.name)
            }
        }
    }

    /**
     * The actual [Logger] instance that all logging functions are delegated to.
     *
     * Attempts to find a reasonable logger name to use by [LoggerName] annotation or [KClass.simpleName].
     */
    val logger: Logger get() = instance.value

    fun trace(message: () -> String) {
        if (logger.isTraceEnabled) {
            logger.trace(message())
        }
    }

    fun debug(message: () -> String) {
        if (logger.isDebugEnabled) {
            logger.debug(message())
        }
    }

    fun info(message: () -> String) {
        if (logger.isInfoEnabled) {
            logger.info(message())
        }
    }

    fun warn(message: () -> String) {
        if (logger.isWarnEnabled) {
            logger.warn(message())
        }
    }

    fun error(message: () -> String) {
        if (logger.isErrorEnabled) {
            logger.error(message())
        }
    }

    fun trace(throwable: Throwable, message: () -> String) {
        if (logger.isTraceEnabled) {
            logger.trace(message(), throwable)
        }
    }

    fun debug(throwable: Throwable, message: () -> String) {
        if (logger.isDebugEnabled) {
            logger.debug(message(), throwable)
        }
    }

    fun info(throwable: Throwable, message: () -> String) {
        if (logger.isInfoEnabled) {
            logger.info(message(), throwable)
        }
    }

    fun warn(throwable: Throwable, message: () -> String) {
        if (logger.isWarnEnabled) {
            logger.warn(message(), throwable)
        }
    }

    fun error(throwable: Throwable, message: () -> String) {
        if (logger.isErrorEnabled) {
            logger.error(message(), throwable)
        }
    }
}
