package ca.deprecatedlogic.debate.parsing

import ca.deprecatedlogic.debate.argument.Argument
import ca.deprecatedlogic.debate.argument.FlagArgument
import ca.deprecatedlogic.debate.argument.NamedArgument
import ca.deprecatedlogic.debate.argument.PositionalArgument
import ca.deprecatedlogic.debate.exception.InvalidFormattingException
import ca.deprecatedlogic.debate.exception.MissingArgumentException
import ca.deprecatedlogic.debate.exception.MissingParameterException
import ca.deprecatedlogic.debate.exception.ParsingError
import ca.deprecatedlogic.debate.exception.ParsingException
import ca.deprecatedlogic.debate.option.DoubleOption
import ca.deprecatedlogic.debate.option.FlagOption
import ca.deprecatedlogic.debate.option.FloatOption
import ca.deprecatedlogic.debate.option.IntegerOption
import ca.deprecatedlogic.debate.option.LongOption
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.ParameterOption
import ca.deprecatedlogic.debate.option.PositionalOption
import ca.deprecatedlogic.debate.option.StringOption
import kotlin.system.exitProcess

/**
 * Parses default options.
 *
 * Will terminate the process and print help if an unrecoverable error occurs during parsing.
 */
open class DefaultParser : Parser {
    private val names = Regex("""--(.+)""")
    private val shorts = Regex("""-([a-zA-Z])""")

    override fun parse(arguments: Array<String>, options: Options): Arguments {
        val elements = ArrayList<Argument>()
        val processed = process(options)

        try {
            processed.forEach {
                elements += find(arguments, it)
            }
        }
        catch (exception: ParsingException) {
            usage(exception.message, processed)
        }

        return Arguments(elements)
    }

    /**
     * Retrieves a list of [Option] objects from an [options] object.
     */
    protected open fun process(options: Options): List<Option> {
        try {
            return options::class.java.declaredFields.mapNotNull {
                it.isAccessible = true
                it.get(options) as? Option
            }
        }
        catch (exception: Exception) {
            throw ParsingError(exception)
        }
    }

    /**
     * Parses an argument matching the given [option] from an array of [arguments].
     *
     * @throws MissingArgumentException Thrown if no argument could be found, no default was specified and
     *                                  the argument is specified as mandatory.
     */
    protected open fun find(arguments: Array<String>, option: Option): Argument {
        /**
         * Locates a matching option identifier argument string.
         */
        fun match(input: String): Boolean {
            val match: MatchResult?

            if (option.short == null) {
                match = names.find(input)
            }
            else {
                match = shorts.find(input) ?: names.find(input)
            }

            if (match == null) {
                return false
            }
            else {
                if (option.short == null) {
                    return option.name == match.groupValues[1]
                }
                else {
                    return option.short == match.groupValues[1][0]
                }
            }
        }

        val argument = arguments.singleOrNull(::match)

        when (option) {
            is FlagOption -> {
                if (argument == null) {
                    return FlagArgument(option.name, false)
                }
                else {
                    return FlagArgument(option.name, true)
                }
            }
            is PositionalOption -> return arguments.positionalArgument(argument, option)
            is ParameterOption<*> -> return arguments.namedArgument(argument, option)
            else -> throw ParsingError()
        }
    }

    /**
     * Parses a positional argument matching the given [option] from program arguments.
     */
    private fun Array<String>.positionalArgument(argument: String?, option: PositionalOption): Argument {
        if (argument == null) {
            if (option.required && option.default == null) {
                throw MissingArgumentException()
            }

            return PositionalArgument(option.name, option.default)
        }

        val parameters = ArrayList<String>()

        var index = indexOf(argument) + 1
        var counter = 1

        while (index < size) {
            if (option.maximum != null) {
                if (counter > option.maximum!!) {
                    break
                }
            }

            val next = this[index]

            if (next.matches(names) || next.matches(shorts)) {
                break
            }

            parameters += next
            index++
            counter++
        }

        if (option.minimum != null) {
            if (counter < option.minimum!!) {
                throw MissingParameterException()
            }
        }

        return PositionalArgument(option.name, parameters)
    }

    /**
     * Parses a named argument matching the given [option] from program arguments.
     */
    private fun Array<String>.namedArgument(argument: String?, option: ParameterOption<*>): Argument {
        if (argument == null) {
            if (option.required && option.default == null) {
                throw MissingArgumentException()
            }

            return NamedArgument(option.name, option.default)
        }

        val start = indexOf(argument)
        val next = start + 1

        if (next < size) {
            val input = this[next]
            val parsed = when (option) {
                is IntegerOption -> input.toIntOrNull() ?: throw InvalidFormattingException()
                is LongOption -> input.toLongOrNull() ?: throw InvalidFormattingException()
                is FloatOption -> input.toFloatOrNull() ?: throw InvalidFormattingException()
                is DoubleOption -> input.toDoubleOrNull() ?: throw InvalidFormattingException()
                is StringOption -> input
                else -> throw ParsingError()
            }

            return NamedArgument(option.name, parsed)
        }
        else {
            if (option.required && option.default == null) {
                throw MissingArgumentException()
            }
            else {
                return NamedArgument(option.name, option.default)
            }
        }
    }
    
    /**
     * Prints error and usage information to [System.out].
     */
    protected open fun usage(message: String?, options: List<Option>) {
        println(buildString {
            error(message)
            appendln()
            shortUsage(options)
            appendln()
            longUsage(options)
        })

        exitProcess(1)
    }

    /**
     * Appends error information to receiver.
     */
    private fun StringBuilder.error(message: String?) {
        append("An error occurred during parsing of program arguments")

        if (message == null) {
            appendln('.')
        }
        else {
            append(':')
            append(' ')
            appendln(message)
        }
    }

    /**
     * Appends example usage to receiver.
     */
    private fun StringBuilder.shortUsage(options: List<Option>) {
        append("Usage: ")

        options.forEachIndexed { index, option ->
            with(option) {
                append('-')
                append('-')
                append(name)

                if (short != null) {
                    append(' ')
                    append('[')
                    append('-')
                    append(short!!)
                    append(']')
                }

                if (index != options.size - 1) {
                    append(' ')
                }
            }
        }

        appendln()
    }

    /**
     * Appends full option information to receiver.
     */
    private fun StringBuilder.longUsage(options: List<Option>) {
        options.forEach {
            with(it) {
                append('-')
                append('-')
                append(name)

                if (short != null) {
                    append(',')
                    append(' ')
                    append('-')
                    append(short!!)
                }

                if (this is PositionalOption) {
                    append('{')

                    if (minimum != null) {
                        append(minimum!!)
                    }

                    append('}')

                    if (maximum != null) {
                        append('{')
                        append(maximum!!)
                        append('}')
                    }
                }

                if (this is ParameterOption<*>) {
                    if (default != null) {
                        append(' ')
                        append('(')
                        append(default!!)
                        append(')')
                    }
                }

                if (help != null) {
                    append(' ')
                    append(':')
                    append(' ')
                    append(help!!)
                }

                appendln()
            }
        }
    }
}
