package ca.deprecatedlogic.debate.parsing

import ca.deprecatedlogic.debate.argument.Argument
import ca.deprecatedlogic.debate.argument.FlagArgument
import ca.deprecatedlogic.debate.argument.NamedArgument
import ca.deprecatedlogic.debate.argument.PositionalArgument
import ca.deprecatedlogic.debate.exception.ParsingError
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.PositionalOption
import kotlin.reflect.KProperty

/**
 * A collection of parsed program arguments, with methods to retrieve parsed argument data.
 */
class Arguments(delegate: List<Argument>) : List<Argument> by delegate {
    /**
     * Returns **true** if the referenced flag is present in the program arguments,
     * **false** if it is not.
     */
    fun flag(property: KProperty<Option>) = parseNamed<Boolean> { property.call() }

    /**
     * Returns **true** if the referenced flag is present in the program arguments,
     * **false** if it is not.
     */
    @Suppress("unused")
    fun flag(supplier: OptionSupplier) = parseNamed<Boolean> { supplier.get() }

    /**
     * Returns the positional parameters, if any, for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun positional(property: KProperty<Option>) = parsePositional { property.call() }

    /**
     * Returns the positional parameters, if any, for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun positional(supplier: OptionSupplier) = parsePositional { supplier.get() }

    /**
     * Returns the [Int] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun integer(property: KProperty<Option>) = parseNamed<Int?> { property.call() }

    /**
     * Returns the [Int] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun integer(supplier: OptionSupplier) = parseNamed<Int?> { supplier.get() }

    /**
     * Returns the [Long] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun long(property: KProperty<Option>) = parseNamed<Long?> { property.call() }

    /**
     * Returns the [Long] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun long(supplier: OptionSupplier) = parseNamed<Long?> { supplier.get() }

    /**
     * Returns the [Float] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun float(property: KProperty<Option>) = parseNamed<Float?> { property.call() }

    /**
     * Returns the [Float] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun float(supplier: OptionSupplier) = parseNamed<Float?> { supplier.get() }

    /**
     * Returns the [Double] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun double(property: KProperty<Option>) = parseNamed<Double?> { property.call() }

    /**
     * Returns the [Double] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun double(supplier: OptionSupplier) = parseNamed<Double?> { supplier.get() }

    /**
     * Returns the [String] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    fun string(property: KProperty<Option>) = parseNamed<String?> { property.call() }

    /**
     * Returns the [String] value for the provided [Option], or **null**
     * if no such value could be found in the parsed program arguments.
     */
    @Suppress("unused")
    fun string(supplier: OptionSupplier) = parseNamed<String?> { supplier.get() }

    /**
     * Parses a flag or named argument.
     */
    @Suppress("UNCHECKED_CAST")
    private fun <T : Any?> parseNamed(supplier: () -> Option): T {
        val option = supplier()
        val match = single { it.name == option.name }

        return when (match) {
            is FlagArgument -> match.value as T
            is NamedArgument -> match.value as T
            else -> throw ParsingError()
        }
    }

    /**
     * Parses a positional argument.
     */
    private fun parsePositional(supplier: () -> Option): List<String>? {
        val option = supplier() as PositionalOption
        val match = single { it.name == option.name } as PositionalArgument

        return match.parameters
    }
}
