package ca.deprecatedlogic.debate.parsing

import ca.deprecatedlogic.debate.argument.Argument
import ca.deprecatedlogic.debate.argument.FlagArgument
import ca.deprecatedlogic.debate.argument.NamedArgument
import ca.deprecatedlogic.debate.argument.PositionalArgument
import ca.deprecatedlogic.debate.exception.InvalidFormattingException
import ca.deprecatedlogic.debate.exception.MissingArgumentException
import ca.deprecatedlogic.debate.exception.ParsingError
import ca.deprecatedlogic.debate.exception.ParsingException
import ca.deprecatedlogic.debate.option.DoubleOption
import ca.deprecatedlogic.debate.option.FlagOption
import ca.deprecatedlogic.debate.option.FloatOption
import ca.deprecatedlogic.debate.option.IntegerOption
import ca.deprecatedlogic.debate.option.LongOption
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.ParameterOption
import ca.deprecatedlogic.debate.option.PositionalOption
import ca.deprecatedlogic.debate.option.StringOption
import kotlin.reflect.full.memberProperties
import kotlin.system.exitProcess

/**
 * Parses default options.
 *
 * Will terminate the process and print help if an unrecoverable error occurs during parsing.
 */
open class DefaultParser : Parser {
    private val names = Regex("""--(.+)""")
    private val shorts = Regex("""-[a-zA-Z]""")

    override fun parse(arguments: Array<String>, options: Options): Arguments {
        val elements = ArrayList<Argument>()
        val properties = options::class.memberProperties // Unsorted
        val filtered = properties.mapNotNull {
            it.call(options) as? Option<*>
        }

        try {
            filtered.forEach {
                elements += find(arguments, it)
            }
        }
        catch (exception: ParsingException) {
            usage(exception.message, filtered)
        }

        return Arguments(elements)
    }

    /**
     * Parses an argument matching the given [option] from an array of [arguments].
     *
     * @throws MissingArgumentException Thrown if no argument could be found, no default was specified and the
     *                                  the argument is specified as mandatory.
     */
    private fun find(arguments: Array<String>, option: Option<*>): Argument {
        /**
         * Locates a matching option identifier argument string.
         */
        fun match(input: String): Boolean {
            val match: MatchResult?

            if (option.short == null) {
                match = names.find(input)
            }
            else {
                match = shorts.find(input) ?: names.find(input)
            }

            if (match == null) {
                return false
            }
            else {
                return when (match.groups.size) {
                    2 -> option.name == match.groupValues[1]
                    1 -> true
                    else -> throw ParsingError()
                }
            }
        }

        val element = arguments.singleOrNull(::match)

        if (option is FlagOption) {
            if (element == null) {
                return FlagArgument(option.name, false)
            }
            else {
                return FlagArgument(option.name, true)
            }
        }
        else if (option is PositionalOption) {
            if (element == null) {
                if (option.required && option.default == null) {
                    throw MissingArgumentException()
                }

                return PositionalArgument(option.name, option.default)
            }

            val parameters = ArrayList<String>()

            var index = arguments.indexOf(element) + 1

            while (index < arguments.size) {
                val next = arguments[index]

                if (next.matches(names) || next.matches(shorts)) {
                    break
                }

                parameters += next
                index++
            }

            return PositionalArgument(option.name, parameters)

        }
        else if (option is ParameterOption<*>) {
            if (element == null) {
                if (option.required && option.default == null) {
                    throw MissingArgumentException()
                }

                return NamedArgument(option.name, option.default)
            }

            val start = arguments.indexOf(element)
            val next = start + 1

            if (next < arguments.size) {
                val argument = arguments[next]
                val parsed = when (option) {
                    is IntegerOption -> argument.toIntOrNull() ?: throw InvalidFormattingException()
                    is LongOption -> argument.toLongOrNull() ?: throw InvalidFormattingException()
                    is FloatOption -> argument.toFloatOrNull() ?: throw InvalidFormattingException()
                    is DoubleOption -> argument.toDoubleOrNull() ?: throw InvalidFormattingException()
                    is StringOption -> argument
                    else -> throw ParsingError()
                }

                return NamedArgument(option.name, parsed)
            }
            else {
                if (option.required && option.default == null) {
                    throw MissingArgumentException()
                }
                else {
                    return NamedArgument(option.name, option.default)
                }
            }
        }
        else {
            throw ParsingError()
        }
    }

    /**
     * Prints error and usage information to [System.out].
     */
    private fun usage(message: String?, options: List<Option<*>>) {
        println(buildString {
            append("An error occurred during parsing of program arguments")

            if (message == null) {
                appendln('.')
            }
            else {
                append(':')
                append(' ')
                appendln(message)
            }

            appendln()
            append("Usage: ")

            options.forEachIndexed { index, option ->
                with(option) {
                    append('-')
                    append('-')
                    append(name)

                    if (short != null) {
                        append(' ')
                        append('[')
                        append('-')
                        append(short!!)
                        append(']')
                    }

                    if (index != options.size - 1) {
                        append(' ')
                    }
                }
            }

            appendln()
            appendln()

            options.forEach {
                with(it) {
                    append('-')
                    append('-')
                    append(name)

                    if (short != null) {
                        append(',')
                        append(' ')
                        append('-')
                        append(short!!)
                    }

                    if (this is PositionalOption) {
                        append('{')

                        if (minimum != null) {
                            append(minimum!!)
                        }

                        append('}')

                        if (maximum != null) {
                            append('{')
                            append(maximum!!)
                            append('}')
                        }
                    }

                    if (this is ParameterOption<*>) {
                        if (default != null) {
                            append(' ')
                            append('(')
                            append(default!!)
                            append(')')
                        }
                    }

                    if (help != null) {
                        append(' ')
                        append(':')
                        append(' ')
                        append(help!!)
                    }

                    appendln()
                }
            }
        })

        exitProcess(1)
    }
}
