@file:JvmName("Parsing")

package ca.deprecatedlogic.debate

import ca.deprecatedlogic.debate.argument.Argument
import ca.deprecatedlogic.debate.argument.Arguments
import ca.deprecatedlogic.debate.argument.Parsed
import ca.deprecatedlogic.debate.argument.Token
import ca.deprecatedlogic.debate.exception.InvalidFormattingException
import ca.deprecatedlogic.debate.exception.MissingArgumentException
import ca.deprecatedlogic.debate.exception.ParsingError
import ca.deprecatedlogic.debate.exception.ParsingException
import ca.deprecatedlogic.debate.option.DoubleOption
import ca.deprecatedlogic.debate.option.FlagOption
import ca.deprecatedlogic.debate.option.FloatOption
import ca.deprecatedlogic.debate.option.IntegerOption
import ca.deprecatedlogic.debate.option.LongOption
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.StringOption
import java.util.LinkedList
import kotlin.reflect.full.memberProperties

private val names = Regex("""^--(.+)$""")
private val shorts = Regex("""^-[a-zA-Z]$""")

/**
 * Parses program [arguments] using the provided expected argument [options] template.
 */
@Suppress("UNCHECKED_CAST")
fun parse(arguments: Array<String>, options: Options): Parsed {
    val elements = ArrayList<Token>()
    val properties = options::class.memberProperties // Unsorted
    val filtered = properties.mapNotNull {
        it.call(options) as? Option<*>
    }

    try {
        for (option in filtered) {
            elements += find(arguments, option)
        }
    }
    catch (exception: ParsingException) {
        val message = buildString {
            append("An error occurred during parsing of program arguments: ")
            appendln(exception.message)
            appendln()
            appendln("Usage:")
            appendln()

            filtered.forEach {
                with(it) {
                    append('-')
                    append('-')
                    append(name)

                    if (short != null) {
                        val flag = short!!

                        append(',')
                        append(' ')
                        append('-')
                        append(flag)
                    }

                    if (defaults != null) {
                        val defaults = defaults!!

                        append(' ')
                        append(defaults)
                    }

                    if (help != null) {
                        val help = help!!

                        append(' ')
                        append(':')
                        append(' ')
                        append(help)
                    }
                    
                    appendln()
                }
            }
        }

        println(message)
    }

    return Parsed(elements)
}

private fun find(arguments: Array<String>, option: Option<*>): Token {
    /**
     * Locates a matching option identifier argument string.
     */
    fun match(input: String): Boolean {
        val match: MatchResult?

        if (option.short == null) {
            match = names.find(input)
        }
        else {
            match = shorts.find(input) ?: names.find(input)
        }

        if (match == null) {
            return false
        }
        else {
            if (match.groups.size > 1) {
                val (value) = match.destructured

                return option.name == value
            }
            else {
                return true
            }
        }
    }

    val element = arguments.singleOrNull(::match)

    if (option is FlagOption) {
        if (element == null) {
            return Argument(option.name, false)
        }
        else {
            return Argument(option.name, true)
        }
    }
    else if (element == null) {
        if (option.required && option.defaults == null) {
            throw MissingArgumentException()
        }

        return Arguments(option.name, option.defaults)
    }
    else {
        val start = arguments.indexOf(element) + 1
        val list = next(arguments, start)

        /**
         * Parses a value from the given [option] if possible.
         */
        fun convert(input: String): Any {
            return when (option) {
                is IntegerOption -> input.toIntOrNull() ?: throw InvalidFormattingException()
                is LongOption -> input.toLongOrNull() ?: throw InvalidFormattingException()
                is FloatOption -> input.toFloatOrNull() ?: throw InvalidFormattingException()
                is DoubleOption -> input.toDoubleOrNull() ?: throw InvalidFormattingException()
                is StringOption -> input
                else -> throw ParsingError()
            }
        }

        val parameters = list.map(::convert)

        return Arguments(option.name, parameters)
    }
}

/**
 * Collates argument parameters until the next argument is found.
 */
private fun next(arguments: Array<String>, start: Int): List<String> {
    val list = arguments.slice(start until arguments.size)
    val queue = LinkedList(list)
    val elements = ArrayList<String>()

    while (true) {
        val next = queue.peek() ?: break
        val matches = next.matches(names) || next.matches(shorts)

        if (matches) {
            break
        }

        elements += queue.poll()
    }

    return elements
}
