package ca.deprecatedlogic.debate

import ca.deprecatedlogic.debate.option.DoubleOption
import ca.deprecatedlogic.debate.option.FlagOption
import ca.deprecatedlogic.debate.option.FloatOption
import ca.deprecatedlogic.debate.option.IntegerOption
import ca.deprecatedlogic.debate.option.LongOption
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.StringOption

/**
 * An implementing object or class serves as a declaration of how your
 * program arguments are to be interpreted.
 */
abstract class Options {
    /**
     * Expect a new [Boolean] program argument.
     */
    fun flag(name: String) = FlagOption(name)

    /**
     * Expect a new [Int] program argument.
     */
    fun integer(name: String) = IntegerOption(name)

    /**
     * Expect a new [Long] program argument.
     */
    fun long(name: String) = LongOption(name)

    /**
     * Expect a new [Float] program argument.
     */
    fun float(name: String) = FloatOption(name)

    /**
     * Expect a new [Double] program argument.
     */
    fun double(name: String) = DoubleOption(name)

    /**
     * Expect a new [String] program argument.
     */
    fun string(name: String) = StringOption(name)

    /**
     * Use this method to set a short-form character for your option.
     */
    fun <T : Any> Option<T>.short(flag: Char): Option<T> {
        this.short = flag
        return this
    }

    /**
     * Use this method to set a help message containing a description
     * of the intended usage and effects of the option.
     */
    fun <T : Any> Option<T>.help(help: String): Option<T> {
        this.help = help
        return this
    }

    /**
     * Use this method to set optional default values that are used if
     * no value is present in program arguments.
     */
    fun <T : Any> Option<T>.defaults(vararg defaults: T): Option<T> {
        this.defaults = defaults.toList()
        return this
    }

    /**
     * Use this method to indicate that a program argument must be present.
     */
    fun <T : Any> Option<T>.required(): Option<T> {
        this.required = true
        return this
    }
}
