@file:JvmName("Parsing")

package ca.deprecatedlogic.debate

import ca.deprecatedlogic.debate.argument.Argument
import ca.deprecatedlogic.debate.argument.Arguments
import ca.deprecatedlogic.debate.argument.Parsed
import ca.deprecatedlogic.debate.argument.Token
import ca.deprecatedlogic.debate.exception.InvalidFormattingException
import ca.deprecatedlogic.debate.exception.MissingArgumentException
import ca.deprecatedlogic.debate.exception.ParsingError
import ca.deprecatedlogic.debate.exception.ParsingException
import ca.deprecatedlogic.debate.option.BooleanOption
import ca.deprecatedlogic.debate.option.DoubleOption
import ca.deprecatedlogic.debate.option.FloatOption
import ca.deprecatedlogic.debate.option.IntegerOption
import ca.deprecatedlogic.debate.option.LongOption
import ca.deprecatedlogic.debate.option.Option
import ca.deprecatedlogic.debate.option.StringOption
import java.util.LinkedList
import kotlin.reflect.full.memberProperties

private val names = Regex("""^--(.+)$""")
private val flags = Regex("""^-[a-zA-Z]$""")

/**
 * Parses program [arguments] using the provided expected argument [options] template.
 */
@Suppress("UNCHECKED_CAST")
fun parse(arguments: Array<String>, options: Options): Parsed {
    val elements = ArrayList<Token>()
    val properties = options::class.memberProperties // Unsorted
    val filtered = properties.mapNotNull {
        it.call(options) as? Option<*>
    }

    try {
        for (option in filtered) {
            /**
             * Locates a matching option identifier argument string.
             */
            fun match(input: String): Boolean {
                val match: MatchResult?

                if (option.flag == null) {
                    match = names.find(input)
                }
                else {
                    match = flags.find(input) ?: names.find(input)
                }

                if (match == null) {
                    return false
                }
                else {
                    if (match.groups.size > 1) {
                        val (value) = match.destructured

                        return option.name == value
                    }
                    else {
                        return true
                    }
                }
            }

            val element = arguments.singleOrNull(::match)

            if (option is BooleanOption) {
                if (element == null) {
                    elements += Argument(option.name, false)
                }
                else {
                    elements += Argument(option.name, true)
                }
            }
            else if (element == null) {
                if (option.required && option.defaults == null) {
                    throw MissingArgumentException()
                }

                elements += Arguments(option.name, option.defaults)
            }
            else {
                val start = arguments.indexOf(element) + 1
                val list = next(arguments, start)

                fun convert(input: String): Any {
                    return when (option) {
                        is IntegerOption -> input.toIntOrNull() ?: throw InvalidFormattingException()
                        is LongOption -> input.toLongOrNull() ?: throw InvalidFormattingException()
                        is FloatOption -> input.toFloatOrNull() ?: throw InvalidFormattingException()
                        is DoubleOption -> input.toDoubleOrNull() ?: throw InvalidFormattingException()
                        is StringOption -> input
                        else -> throw ParsingError()
                    }
                }

                val parameters = list.map(::convert)

                elements += Arguments(option.name, parameters)
            }
        }
    }
    catch (exception: ParsingException) {
        val message = buildString {
            append("An error occurred during parsing of program arguments.")
            TODO("append usage information")
        }

        System.err.print(message)
    }

    return Parsed(elements)
}

/**
 * Parses
 */
private fun next(arguments: Array<String>, start: Int): List<String> {
    val list = arguments.slice(start until arguments.size)
    val queue = LinkedList(list)
    val elements = ArrayList<String>()

    while (true) {
        val next = queue.peek() ?: break
        val matches = next.matches(names) || next.matches(flags)

        if (matches) {
            break
        }

        elements += queue.poll()
    }

    return elements
}

//private fun process(arguments: Array<String>, option: Option<*>): Token? {
//    if (arguments.isEmpty()) {
//        return null
//    }
//    
//    val list = arguments.asList()
//    val queue = LinkedList<String>(list)
//    
//    when(option) {
//        is BooleanOption -> return boolean(option, queue.poll())
//        is IntegerOption -> return integer(option, queue)
//        is LongOption -> return long(option, queue)
//        is FloatOption -> return float(option, queue)
//        is DoubleOption -> return double(option, queue)
//        is StringOption -> return string(option, queue)
//        else -> throw ParsingError("Unknown option type.")
//    }
//}
//
//private fun boolean(option: BooleanOption, argument: String?): Token? {
//    if (argument == null) {
//        return null
//    }
//    
//    val match: MatchResult?
//
//    if (option.flag == null) {
//        match = names.find(argument)
//    }
//    else {
//        match = flags.find(argument)
//    }
//
//    val value: Boolean
//
//    if (match == null) {
//        value = false
//    }
//    else {
//        val (name) = match.destructured
//
//        value = name == option.name
//    }
//
//    return Argument(option.name, value)
//}
//
//private fun integer(option: IntegerOption, queue: Queue<String>): Token? {
//    if (queue.isEmpty()) {
//        return null
//    }
//    
//    val parameters = next(option, queue) ?: return null
//    val integers = parameters.map { it.toInt() }
//
//    return Arguments(option.name, integers)
//}
//
//private fun long(option: LongOption, queue: Queue<String>): Token? {
//    if (queue.isEmpty()) {
//        return null
//    }
//    
//    val parameters = next(option, queue) ?: return null
//    val integers = parameters.map { it.toLong() }
//
//    return Arguments(option.name, integers)
//}
//
//private fun float(option: FloatOption, queue: Queue<String>): Token? {
//    if (queue.isEmpty()) {
//        return null
//    }
//    
//    val parameters = next(option, queue) ?: return null
//    val integers = parameters.map { it.toFloat() }
//
//    return Arguments(option.name, integers)
//}
//
//private fun double(option: DoubleOption, queue: Queue<String>): Token? {
//    if (queue.isEmpty()) {
//        return null
//    }
//    
//    val parameters = next(option, queue) ?: return null
//    val integers = parameters.map { it.toDouble() }
//
//    return Arguments(option.name, integers)
//}
//
//private fun string(option: StringOption, queue: Queue<String>): Token? {
//    if (queue.isEmpty()) {
//        return null
//    }
//    
//    val parameters = next(option, queue) ?: return null
//
//    return Arguments(option.name, parameters)
//}
//
