package by.shostko.android.rxlifecycle;

import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.LifecycleOwner;
import io.reactivex.*;
import io.reactivex.functions.Action;
import io.reactivex.functions.BiFunction;
import io.reactivex.functions.Function;
import io.reactivex.functions.Predicate;
import org.reactivestreams.Publisher;

import java.util.concurrent.Callable;

@SuppressWarnings({"unused", "WeakerAccess"})
public final class RxLifecycle<T> implements FlowableTransformer<T, T>,
        ObservableTransformer<T, T>, MaybeTransformer<T, T>, CompletableTransformer {

    @NonNull
    private final Lifecycle lifecycle;

    @NonNull
    private final Lifecycle.State atLeastState;

    @NonNull
    private final BiFunction<T, Lifecycle.State, Pair<T, Lifecycle.State>> pairCreation = new BiFunction<T, Lifecycle.State, Pair<T, Lifecycle.State>>() {
        @Override
        public Pair<T, Lifecycle.State> apply(T t, Lifecycle.State state) {
            return Pair.create(t, state);
        }
    };

    @NonNull
    private final Predicate<Pair<T, Lifecycle.State>> whilePredicate = new Predicate<Pair<T, Lifecycle.State>>() {
        @Override
        public boolean test(Pair<T, Lifecycle.State> pair) {
            return pair.second.isAtLeast(Lifecycle.State.INITIALIZED);
        }
    };

    @NonNull
    private final Predicate<Pair<T, Lifecycle.State>> filterPredicate = new Predicate<Pair<T, Lifecycle.State>>() {
        @Override
        public boolean test(Pair<T, Lifecycle.State> pair) {
            return pair.second.isAtLeast(atLeastState);
        }
    };

    @NonNull
    private final Function<Pair<T, Lifecycle.State>, T> valueExtractor = new Function<Pair<T, Lifecycle.State>, T>() {
        @Override
        public T apply(Pair<T, Lifecycle.State> pair) {
            return pair.first;
        }
    };

    public RxLifecycle(@NonNull LifecycleOwner lifecycleOwner) {
        this.lifecycle = lifecycleOwner.getLifecycle();
        this.atLeastState = Lifecycle.State.STARTED;
    }

    public RxLifecycle(@NonNull LifecycleOwner lifecycleOwner, @NonNull Lifecycle.State atLeastState) {
        this.lifecycle = lifecycleOwner.getLifecycle();
        this.atLeastState = atLeastState;
    }

    public RxLifecycle(@NonNull Lifecycle lifecycle) {
        this.lifecycle = lifecycle;
        this.atLeastState = Lifecycle.State.STARTED;
    }

    public RxLifecycle(@NonNull Lifecycle lifecycle, @NonNull Lifecycle.State atLeastState) {
        this.lifecycle = lifecycle;
        this.atLeastState = atLeastState;
    }

    @Override
    @NonNull
    public Flowable<T> apply(@NonNull Flowable<T> upstream) {
        Flowable<Lifecycle.State> statesFlowable = states(lifecycle);
        return Flowable.combineLatest(upstream, statesFlowable, pairCreation)
                .takeWhile(whilePredicate)
                .filter(filterPredicate)
                .map(valueExtractor);
    }

    @Override
    @NonNull
    public ObservableSource<T> apply(@NonNull Observable<T> upstream) {
        Observable<Lifecycle.State> stateObservable = states(lifecycle).toObservable();
        return Observable.combineLatest(upstream, stateObservable, pairCreation)
                .takeWhile(whilePredicate)
                .filter(filterPredicate)
                .map(valueExtractor);
    }

    @Override
    @NonNull
    public MaybeSource<T> apply(@NonNull Maybe<T> upstream) {
        return apply(upstream.toFlowable()).firstElement();
    }

    @Override
    @NonNull
    public CompletableSource apply(@NonNull Completable upstream) {
        return apply(upstream.<T>toFlowable()).ignoreElements();
    }

    @NonNull
    public static Flowable<Lifecycle.Event> events(@NonNull final Lifecycle lifecycle) {
        return events(lifecycle, BackpressureStrategy.LATEST);
    }

    @NonNull
    public static Flowable<Lifecycle.Event> events(@NonNull final Lifecycle lifecycle,
                                                   @NonNull BackpressureStrategy backpressureStrategy) {
        final LifecycleEventObserver observer = new LifecycleEventObserver();
        return Flowable.create(new FlowableOnSubscribe<Lifecycle.Event>() {
            @Override
            public void subscribe(FlowableEmitter<Lifecycle.Event> emitter) {
                observer.setEmitter(emitter);
                lifecycle.addObserver(observer);
            }
        }, backpressureStrategy)
                .doFinally(new Action() {
                    @Override
                    public void run() {
                        lifecycle.removeObserver(observer);
                    }
                })
                .startWith(Flowable.defer(new Callable<Publisher<? extends Lifecycle.Event>>() {
                    @Override
                    public Publisher<? extends Lifecycle.Event> call() {
                        switch (lifecycle.getCurrentState()) {
                            case RESUMED:
                                return Flowable.just(Lifecycle.Event.ON_RESUME);
                            case DESTROYED:
                                return Flowable.just(Lifecycle.Event.ON_DESTROY);
                            default:
                                return Flowable.empty();
                        }
                    }
                }));
    }

    @NonNull
    public static Flowable<Lifecycle.State> states(@NonNull final Lifecycle lifecycle) {
        return states(lifecycle, BackpressureStrategy.LATEST);
    }

    @NonNull
    public static Flowable<Lifecycle.State> states(@NonNull final Lifecycle lifecycle,
                                                   @NonNull BackpressureStrategy backpressureStrategy) {
        final LifecycleStateObserver observer = new LifecycleStateObserver(lifecycle);
        return Flowable.create(new FlowableOnSubscribe<Lifecycle.State>() {
            @Override
            public void subscribe(FlowableEmitter<Lifecycle.State> emitter) {
                observer.setEmitter(emitter);
                lifecycle.addObserver(observer);
            }
        }, backpressureStrategy)
                .doFinally(new Action() {
                    @Override
                    public void run() {
                        lifecycle.removeObserver(observer);
                    }
                })
                .startWith(Flowable.fromCallable(new Callable<Lifecycle.State>() {
                    @Override
                    public Lifecycle.State call() {
                        return lifecycle.getCurrentState();
                    }
                }));
    }
}
