(ns singularity.settings

  "Parse the singularity.json configuration file and manage settings across the
  entire application.  Also understands the current state of the server, whether
  it is in development, test, or production mode."

  (:refer-clojure :exclude (for))
  (:use [clojure.data.json :only (read-json)]
        [clojure.java.io :only (resource)])
  (:require [clojure.string :as str]
            [clojure.java.io :as io]))

(def ^:const FILENAME "singularity.json")

(defonce ^:dynamic *settings* nil)

(defn file?
  "Does the given file exist?"
  [file]
  (.exists (io/file file)))

(defn load-settings
  "Create a reader for the correct settings file.  Checks to see if there is a
  file /etc/brilliantarc/singularity.json.  If it's not there, it tries to load
  it from the classpath."
  []
  (let [full-settings (str "/etc/brilliantarc/" FILENAME)]
    (if (file? full-settings)
      (read-json (io/reader full-settings))
      (read-json (io/reader (resource FILENAME))))))

(defn env
  "Load the settings for the given environment and cache them.  If the
  environment is not provided, the singularity.env system property is used.
  If that is not provided, defaults to \"development\"."
  [& [environment]]
  (alter-var-root #'*settings* (fn [_]
                                 (let [e (or environment "development")]
                                   (get (load-settings) (keyword e))))))

(defn for
  "Look up a setting for the current environment, i.e. the runtime environment
  defined in the singularity.env system property.

  You can \"deep dive\" for values within objects in the JSON settings using
  dot notation.  For example, if you have settings like this:

    {
      \"rabbitmq\": {
        \"host\": \"localhost\"
      }
    }

  you can reference the RabbitMQ host by calling:

    (settings/for :rabbitmq.host)
  "
  ([settings property]
    (if (and settings property)
      (let [[key rest] (str/split (name property) #"\.", 2)
            value (settings (keyword key))]
        (if rest
          (recur value rest)
          value))))

  ([property] (for (or *settings* (env)) property)))
