package br.ufrgs.cpd.coresdk.token

import android.annotation.SuppressLint
import android.content.Context
import android.provider.Settings
import br.ufrgs.cpd.coresdk.R
import br.ufrgs.cpd.coresdk.UfrgsSdk
import br.ufrgs.cpd.coresdk.data.UfrgsDataManager
import br.ufrgs.cpd.coresdk.data.models.UfrgsToken
import br.ufrgs.cpd.coresdk.network.ApiBuilder
import br.ufrgs.cpd.coresdk.network.ApiInterface
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import retrofit2.HttpException
import kotlin.Exception


class UfrgsTokenManager {

    private val clientId = UfrgsSdk.clientId
    private val clientSecret = UfrgsSdk.clientSecret
    private val scope = UfrgsSdk.scope
    private val grantType = UfrgsSdk.grantType

    private val service by lazy { ApiBuilder.retrofit().create(ApiInterface::class.java) }

    fun isLoggedIn(context: Context): Boolean {
        val token = UfrgsDataManager().getToken(context)
        return token != null
    }

    fun getToken(context: Context): UfrgsToken? {
        return UfrgsDataManager().getToken(context)
    }

    fun getAccessToken(context: Context): String? {
        return UfrgsDataManager().getToken(context)?.accessToken
    }

    fun getBearerToken(context: Context): String? {
        return "Bearer ${UfrgsDataManager().getToken(context)?.accessToken}"
    }

    @SuppressLint("HardwareIds")
    fun logout(context: Context, callback: OnLogoutComplete) {

        val tokenHeader = getBearerToken(context)

        if (tokenHeader == null) {
            callback.onLogoutReady(false)
            return
        }

        val androidId = Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID)
        val deferred = service.logoutFromAPI(tokenHeader, androidId)

        GlobalScope.launch(Dispatchers.Main) {
            try {
                withContext(Dispatchers.IO) { deferred.await() }
                UfrgsDataManager().deleteToken(context)
                callback.onLogoutReady(true)
            } catch (e: Exception) {
                callback.onLogoutReady(false)
            }
        }
    }


    fun requestNewToken(context: Context, username: String, password: String, callback: OnTokenListener) {
        GlobalScope.launch(Dispatchers.Main) {
            val tokenDeferred = service.getToken(clientId, clientSecret, grantType, username, password, scope)

            try {
                val response = withContext(Dispatchers.IO) { tokenDeferred.await() }
                val token = response.data

                val ufrgsToken = UfrgsToken.newInstance(token)
                UfrgsDataManager().saveToken(context, ufrgsToken)

                callback.onTokenReady(ufrgsToken)
            } catch (e: Exception) {
                val errorMessage = (e as HttpException).response().errorBody()?.string()
                        ?: "unknown error"
                callback.onError(context.getString(R.string.login_error))
            }
        }
    }

    @SuppressLint("HardwareIds")
    fun registerGcmToken(context: Context, gcmToken: String) {
        val tokenHeader = getBearerToken(context) ?: return
        val androidId = Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID)

        GlobalScope.launch(Dispatchers.Main) {
            val deferred = service.registerGcmDevice(tokenHeader, gcmToken, androidId)
            try {
                withContext(Dispatchers.IO) { deferred.await() }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }
}