@file:Suppress("UNCHECKED_CAST")

package kotlinjs.runtime.container

import kotlinjs.runtime.JsType
import kotlinjs.runtime.extension.typeOf


/**
 * Created by danfma on 12/05/16.
 */
@Suppress("unused")
class Container {
    private val registries = hashMapOf<String, ServiceRegistry<*>>()
    private val instances = hashMapOf<String, Any>()

    private fun <T : Any> registerInstance(name: String, instance: T): T {
        instances.put(name, instance)

        return instance
    }

    fun <T : Any> registerSingleton(name: String, type: JsType<T>, factory: Function<T>, vararg dependencies: ServiceRegistry<out Any>): ServiceRegistry<T> {
        val registry = ServiceRegistry(name, true, type, dependencies) {
            registerInstance(name, resolveInstance(factory, dependencies))
        }

        registries.put(name, registry)

        return registry
    }

    fun <T : Any> registerTransient(name: String, type: JsType<T>, factory: Function<T>, vararg dependencies: ServiceRegistry<out Any>): ServiceRegistry<T> {
        val registry = ServiceRegistry(name, true, type, dependencies) {
            resolveInstance(factory, dependencies)
        }

        registries.put(name, registry)

        return registry
    }

    private fun <T> resolveInstance(factory: Function<T>, dependencies: Array<out ServiceRegistry<out Any>>): T {
        val dependencyInstances = dependencies.map { dep -> resolveInstance(dep) }.toTypedArray()
        val func: dynamic = factory

        return func.apply(null, dependencyInstances)
    }


    @Suppress("UNCHECKED_CAST")
    private fun <T : Any> resolveInstance(registry: ServiceRegistry<T>): T {
        if (!registry.isSingleton) {
            return registry.factory()
        }

        return instances.getOrPut(registry.name, { registry.factory() }) as T
    }

    fun <T : Any> resolveByName(name: String): T {
        val registry = registries[name] ?: throw Error("Component '$name' was not found.")

        return resolveInstance(registry as ServiceRegistry<T>)
    }

    fun <T : Any> resolveByType(serviceType: JsType<T>): T {
        val registry = registries.values.first { it.type == serviceType }

        return resolveByName(registry.name)
    }

    inline fun <reified T : Any> resolveByType(): T {
        return resolveByType(typeOf<T>())
    }


    inline fun <reified T : Any> singleton(name: String, noinline factory: () -> T): ServiceRegistry<T> {
        return registerSingleton(name, typeOf<T>(), factory)
    }


    inline fun <reified T : Any, TDep1 : Any> singleton(
            name: String,
            dependency: ServiceRegistry<TDep1>,
            noinline factory: (TDep1) -> T): ServiceRegistry<T> {

        return registerSingleton(name, typeOf<T>(), factory, dependency)
    }

    inline fun <reified T : Any, TDep1 : Any, TDep2 : Any> singleton(
            name: String,
            dependency1: ServiceRegistry<TDep1>,
            dependency2: ServiceRegistry<TDep2>,
            noinline factory: (TDep1, TDep2) -> T): ServiceRegistry<T> {

        return registerSingleton(name, typeOf<T>(), factory, dependency1, dependency2)
    }

    inline fun <reified T : Any, TDep1 : Any, TDep2 : Any, TDep3 : Any> singleton(
            name: String,
            dependency1: ServiceRegistry<TDep1>,
            dependency2: ServiceRegistry<TDep2>,
            dependency3: ServiceRegistry<TDep3>,
            noinline factory: (TDep1, TDep2, TDep3) -> T): ServiceRegistry<T> {

        return registerSingleton(name, typeOf<T>(), factory, dependency1, dependency2, dependency3)
    }

    inline fun <reified T : Any> transient(name: String, noinline factory: () -> T): ServiceRegistry<T> {
        return registerTransient(name, typeOf<T>(), factory)
    }


    inline fun <reified T : Any, TDep1 : Any> transient(
            name: String,
            dependency: ServiceRegistry<TDep1>,
            noinline factory: (TDep1) -> T): ServiceRegistry<T> {

        return registerTransient(name, typeOf<T>(), factory, dependency)
    }

    inline fun <reified T : Any, TDep1 : Any, TDep2 : Any> transient(
            name: String,
            dependency1: ServiceRegistry<TDep1>,
            dependency2: ServiceRegistry<TDep2>,
            noinline factory: (TDep1, TDep2) -> T): ServiceRegistry<T> {

        return registerTransient(name, typeOf<T>(), factory, dependency1, dependency2)
    }

    inline fun <reified T : Any, TDep1 : Any, TDep2 : Any, TDep3 : Any> transient(
            name: String,
            dependency1: ServiceRegistry<TDep1>,
            dependency2: ServiceRegistry<TDep2>,
            dependency3: ServiceRegistry<TDep3>,
            noinline factory: (TDep1, TDep2, TDep3) -> T): ServiceRegistry<T> {

        return registerTransient(name, typeOf<T>(), factory, dependency1, dependency2, dependency3)
    }


    fun makeGlobal() {
        instance = this
    }


    companion object {
        lateinit var instance: Container

    }

}
