package kotlinjs.runtime

import kotlinjs.runtime.decorator.ClassDecorator
import kotlinjs.runtime.decorator.addClassDecorators

class TypeInfo<T>(val fullName: String,
                  val simpleName: String,
                  type: JsType<T>) {

    val typePrototype: T
        get() {
            return type.asDynamic().prototype
        }

    var type: JsType<T> = type
        get
        private set

    fun decorate(vararg decorators: ClassDecorator) {
        type = type.addClassDecorators(decorators)
    }

}

inline fun <T> typeInfo(creator: () -> T): TypeInfo<T>? {
    val creatorString: String = js("(creator.toString())")
    val names = getFullTypeName(creatorString)!!
    val type = resolveTypeByResolutionPath<T>(names.localResolutionPath)

    return TypeInfo(names.fullName, names.simpleName, type)
}

@Suppress("NOTHING_TO_INLINE")
inline fun <T> resolveTypeByResolutionPath(path: String): JsType<T> = js("(eval(path))")

fun getFullTypeName(creatorString: String): TypeName? {
    val pattern = Regex("""new (.*)\(""")
    val match = pattern.find(creatorString) ?: return null
    val resolutionPath = match.groupValues[1]
    val fullName = resolutionPath.substring(2)

    val simpleName =
            if (fullName.contains(".")) {
                val index = fullName.lastIndexOf('.')
                fullName.substring(index + 1)
            } else {
                fullName
            }

    return TypeName(resolutionPath, fullName, simpleName)
}

data class TypeName(
        val localResolutionPath: String,
        val fullName: String,
        val simpleName: String)
