package kotlinjs.runtime

import kotlinjs.runtime.decorator.ClassDecorator
import kotlinjs.runtime.decorator.addClassDecorators

class TypeInfo<T>(val fullName: String,
                  val simpleName: String,
                  type: JsType<T>) {

    val typePrototype: T
        get() {
            return type.asDynamic().prototype
        }

    var type: JsType<T> = type
        get
        private set

    fun decorate(vararg decorators: ClassDecorator) {
        type = type.addClassDecorators(decorators)
    }

}

class TypeInfoBuilder<T>(creator: () -> T) {

    val creatorAsString = creator.toString()
    val names = getFullTypeName(creatorAsString)!!
    lateinit var type: JsType<T>

    fun resolve(typeResolver: (String) -> JsType<T>): TypeInfoBuilder<T> {
        type = typeResolver(names.localResolutionPath)
        return this
    }

    fun build(): TypeInfo<T>? {
        return TypeInfo(names.fullName, names.simpleName, type)
    }
}

@Suppress("NOTHING_TO_INLINE")
inline fun <T> typeInfo(noinline creator: () -> T): TypeInfo<T>? {
    val resolver: (String) -> JsType<T> = { resolveTypeByResolutionPath(it) }

    return TypeInfoBuilder(creator)
            .resolve(resolver)
            .build()
}

@Suppress("NOTHING_TO_INLINE")
inline fun <T> resolveTypeByResolutionPath(path: String): JsType<T> = js("(eval(path))")

fun getFullTypeName(creatorString: String): TypeName? {
    val pattern = Regex("""new (.*)\(""")
    val match = pattern.find(creatorString) ?: return null
    val resolutionPath = match.groupValues[1]
    val fullName = resolutionPath.substring(2)

    val simpleName =
            if (fullName.contains(".")) {
                val index = fullName.lastIndexOf('.')
                fullName.substring(index + 1)
            } else {
                fullName
            }

    return TypeName(resolutionPath, fullName, simpleName)
}

data class TypeName(
        val localResolutionPath: String,
        val fullName: String,
        val simpleName: String)
